#ifndef  D0RunIIconeJets_PROTOJET
#define  D0RunIIconeJets_PROTOJET
// ---------------------------------------------------------------------------
// ProtoJet.hpp
//
// Created: 28-JUL-2000 Francois Touze (+ Laurent Duflot)
//
// Purpose: Implements a proto-jet object that is used as input by the 
//   Improved Legacy Cone Algorithm split/merge algo.
//
// Modified:
//    9-Aug-2000  Laurent Duflot
//     + save the initial stable cone ET before split/merge
//    1-May-2007 Lars Sonnenschein
//    extracted from D0 software framework and modified to remove subsequent dependencies 
// ---------------------------------------------------------------------------
// 
#include "ConeJetInfo.hh"
#include <list>
#include <cmath>

#include "inline_maths.hh" //ls

namespace SpartyJet { 
  using namespace SpartyJet::inline_maths;
  using namespace SpartyJet::D0RunIIconeJets_CONEJETINFO;



inline float RD2(float y1,float phi1,float y2,float phi2) 
{
  float dphi= delta_phi(phi1,phi2);
  return (y1-y2)*(y1-y2)+dphi*dphi; 
}

inline float RDelta(float y1,float phi1,float y2,float phi2) 
{
  float dphi= delta_phi(phi1,phi2);
  return sqrt((y1-y2)*(y1-y2)+dphi*dphi); 
}

inline float P2y(float* p4vec) {
  return y(p4vec[3],p4vec[2]);
}

inline float P2phi(float* p4vec) {
  return phi(p4vec[0],p4vec[1]);
}

///////////////////////////////////////////////////////////////////////////////
class ProtoJet {

public :

  ProtoJet(float seedET);
  ProtoJet(float seedET,float y,float phi);
  ProtoJet(const ProtoJet& pj);
  ~ProtoJet() {;}

  void addItem(const Jet* tw); 
  void setJet(float y,float phi,float pT); 
  void updateJet();
  void erase();

  float rapidity() const; 
  float phi() const;
  float pT() const;
  const ConeJetInfo & info() const;
  const std::list<const Jet*>& LItems() const;

  void print(std::ostream &os) const;

  // actions to be taken when the jet is a stable cone
  void NowStable();
  // declare the jet to have been splitted
  void splitted(){_info.splitted();};
  // declare the jet to have been merged
  void merged(){_info.merged();};
protected :

  std::list<const Jet*> _LItems;
  float _y;
  float _phi;
  float _pT;
  ConeJetInfo _info;

};
///////////////////////////////////////////////////////////////////////////////
ProtoJet::ProtoJet(float seedET) : _LItems(), _info(seedET) {
    _y  = 0.0;
    _phi= 0.0;
    _pT = 0.0;
}

ProtoJet::ProtoJet(float seedET,float y,float phi) :  _LItems(), _info(seedET) { 
  _y  = y; 
  _phi= phi;
  _pT = 0.0;
}

ProtoJet::ProtoJet(const ProtoJet& pj): _y(pj._y), 
						    _phi(pj._phi), _pT(pj._pT),
                                                    _info(pj._info)
{ 
  std::list<const Jet*>::const_iterator it;
  for(it = pj._LItems.begin(); it != pj._LItems.end(); ++it) { 
    _LItems.push_back(*it);
  }
}

void ProtoJet::addItem(const Jet* tw) {
  _LItems.push_back(tw);
}

void ProtoJet::setJet(float y,float phi,float pT) {
  _y  = y;
  _phi= phi;
  _pT = pT;
}

void ProtoJet::updateJet() { 
  float p[4] = {0.,0.,0.,0.};
  std::list<const Jet*>::iterator it;
  for(it = _LItems.begin(); it != _LItems.end(); ++it) 
  {
    float pk[4];
    pk[0] = (*it)->px();
    pk[1] = (*it)->py();
    pk[2] = (*it)->pz();
    pk[3] = (*it)->E();

    for ( int i = 0; i < 4 ; ++i) p[i] += pk[i];
  }

  _y = P2y(p);
  _phi = P2phi(p);
  _pT = sqrt(p[0]*p[0] + p[1]*p[1]);
  if ( p[3] < 0. ) _pT = - _pT;

}

void ProtoJet::erase() {
  _LItems.erase(_LItems.begin(),_LItems.end());
  _y  = 0.0;
  _phi= 0.0;
  _pT = 0.0; 
  // _info is not modified in order to keep split/merge history
}

// actions to be taken when the jet is a stable cone
void ProtoJet::NowStable() {
  _info.initialET(_pT);
}

// void ProtoJet::print(std::ostream& os) const {
//   os<<"y phi Et = ("<<_y<<", "<<_phi<<", "<<this->_Et<<")"<<std::endl;
//   os<< " members= " << std::endl;
//   std::list<const Jet*>::const_iterator i;
//   for(i = _LItems.begin(); i != _LItems.end(); ++i)
//     (*i)->print(os);
//   os << std::endl;
// }

inline float ProtoJet::rapidity() const{
  return _y;
}

inline float ProtoJet::phi() const{
  return _phi;
}

inline float ProtoJet::pT() const{
  return _pT;
}
inline const ConeJetInfo & ProtoJet::info() const{
  return _info;
}

inline const std::list<const Jet*>& ProtoJet::LItems() const{
  return _LItems;
}
///////////////////////////////////////////////////////////////////////////////
}  // namespace SpartyJet
#endif
