#include "JetAlgorithm.hh"
#include "JetTool.hh"
#include "CommonUtils.hh"

#include <iostream>



namespace SpartyJet { 

void JetAlgorithm::addTool(JetTool * tool){
  m_toolList.push_back(tool);
}

void JetAlgorithm::addTool_front(JetTool * tool){
  m_toolList.push_front(tool);
}

bool JetAlgorithm::init(){
  jettool_list_t::iterator it = m_toolList.begin();
  jettool_list_t::iterator itE = m_toolList.end();

  for(; it != itE; ++it){
    (*it)->init(&m_mmap);
    m_log << DEBUG << "Initialized tool : " << (*it)->name() << std::endl;
    if(m_measureTime) prepareTime(*it);
    m_doHistory = m_doHistory | (*it)->updateHistory();
  }

  return true;
  
}
bool JetAlgorithm::execute(Jet::jet_list_t &inputJets, JetCollection &outputJets){
  // copy the structure of JetMoments from JetAlgorithm into output collection
  outputJets.get_JetMomentMap()->copy_structure(m_mmap);
  
	// copy Jets completely (not just pointers)
	outputJets.deep_copy(inputJets);
  
	// now outputJets can safely be modified by tools 
  // this will not affect inputJets
	return execute(outputJets);
}

bool JetAlgorithm::execute(JetCollection &inputJets, JetCollection &outputJets){
  // copy the structure of JetMoments from JetAlgorithm into output collection
  outputJets.get_JetMomentMap()->copy_structure(m_mmap);
  
	// copy Jets completely (not just pointers, reassign JetMoments correctly)
	outputJets.deep_copy(inputJets);

  // now outputJets can safely be modified by tools 
  // this will not affect inputJets
	return execute(outputJets);
}
  
bool JetAlgorithm::execute(JetCollection &outputJets){
  // Loop over tools :
  jettool_list_t::iterator it = m_toolList.begin();
  jettool_list_t::iterator itE = m_toolList.end();
  stopwatch watch;
  for(; it != itE; ++it){

    watch.start();
    
    m_log << INFO << "---- " << (*it)->name() << std::endl;
    
		(*it)->execute(outputJets);

    m_log << INFO << "---- " << "remaining jets: "<< outputJets.size() << std::endl;

    // save times if needed :
    float t = watch.stop();
    if(m_measureTime) recordTime(*it, t, outputJets.get_JetMomentMap()); 
  }

  // sort by Et by default
  sort_list_et(outputJets);



  if(m_doHistory){
    // nothing for now
  }

  // testing only
  if(outputJets.has_history()) {
    std::cout <<" end alg, dump history "<<std::endl;
    outputJets.history_dump();
    std::cout <<" end alg, dumped history "<<std::endl;
  }
    

  return true;  
}


bool JetAlgorithm::finalize(){
  jettool_list_t::iterator it = m_toolList.begin();
  jettool_list_t::iterator itE = m_toolList.end();
  
  for(; it != itE; ++it){
    (*it)->finalize();
  }
  return true;    
}

void JetAlgorithm::prepareTime(JetTool *jt){
  std::string name = jt->name();
  if(name == "selector" ) return; // not interesting.
  m_mmap.schedule_event_moment(name+"Time");  
}
void JetAlgorithm::recordTime(JetTool *jt, float t, JetMomentMap * map){
  std::string name = jt->name();
  if(name == "selector" ) return; // not interesting.
  map->set_event_moment(name+"Time",t);
}
}  // namespace SpartyJet
