#ifndef _JETALGORITHM_HH_
#define _JETALGORITHM_HH_
//*****************************************************************
/// \class JetAlgorithm
/// 
/// This class holds a list of JetTools which all together
/// perform a desired jet search
/// In its execute(inputJets,outputJets) method, JetAlgorithm copies inputJet into outputJets
/// then calls sequentially the execute() method of its tools upon outputJets.
///  Thus outputJets finally contains the result of the JetTool sequence.
/// JetAlgorithm also holds a JetMomentMap. This map is passed to the JetTools in the init() method
/// so that the JetTools can fill it when necessary.
//*****************************************************************

#include "JetMomentMap.hh"
#include "Jet.hh"
#include "JetCollection.hh"

#include <list>
#include <string>
#include "CustomMessage.hh"
namespace SpartyJet { 

class JetTool;
typedef std::list<JetTool*> jettool_list_t;

class JetAlgorithm {
public:

  JetAlgorithm(): m_name("noname"), m_measureTime(false), m_doHistory(false){}
  JetAlgorithm(std::string name) : m_name(name), m_measureTime(false) , m_doHistory(false) {m_log.set_name(name);};
  ~JetAlgorithm(){m_mmap.clear();}

  /// Add a JetTool
  /// addition order matters. Must be called before init()
  void addTool(JetTool * tool);

  /// Add a JetTool before any other 
  /// addition order matters. Must be called before init()
  void addTool_front(JetTool * tool);

  /// Initialize this alg and all its JetTools.
  bool init();  

  /// Run the jet tool sequence
  /// @param inputJets is unaltered. 
  /// @param outputJets is passed to execute(JetCollection &outputJet)
  bool execute(Jet::jet_list_t &inputJets, JetCollection &outputJet);  
  
	/// Run the jet tool sequence
  /// @param inputJets is unaltered. 
  /// @param outputJets is passed to execute(JetCollection &outputJet)
  bool execute(JetCollection &inputJets, JetCollection &outputJet);
  
	/// Run the jet tool sequence
  /// @param outputJets is passed to the tools for modification
	bool execute(JetCollection &outputJet);

  /// Finalize this alg and all its JetTools.
  bool finalize();

  std::string name(){return m_name;}
  void set_name(std::string name){m_name = name; m_log.set_name(name);}
  
  JetMomentMap * get_jet_momentMap(){return &m_mmap;}

  /// Allows timing measurement. Must be called before init()
  void doTimeMeasure(){m_measureTime = true;}

  // removes all messages
  void make_silent() {m_log.g_level = ERROR;}

protected:
  
  JetMomentMap m_mmap;  /// This map is only needed to hold structure of JetMomentMap to be computed
                        /// the actual moments will be stored in the JetCollection. 
  std::string m_name;   /// The algo name
  jettool_list_t m_toolList;  /// The list of JetTools 

  
  bool m_measureTime;
  void prepareTime(JetTool *jt);
  void recordTime(JetTool *jt, float t, JetMomentMap* map);

  bool m_doHistory;

  Message m_log;
};

}  // namespace SpartyJet
#endif
