#include <iostream>
#include <iomanip>
#include <ctime> // for poisson minbias random numbers

#include "JetAlgorithm.hh"
#include "JetTool.hh"
#include "InputMaker.hh"
#include "CommonUtils.hh"
#include "GhostBuilder.hh"
#include "JetGlobals.hh"
#include "JetNegEnergyTool.hh"
#include "JetBuilder.hh"

namespace SpartyJet { 
JetBuilder::JetBuilder(MessageLevel ml){
  m_log.set_name("JetBuilder");
  Message::set_message_level(ml);
	m_GeV=1;
  m_MeV=0.001;
	m_needInit = true;

  m_correct_neg_energy = false;
  m_neg_energyTool = new JetNegEnergyTool("NegEnergyTool");
  m_updateOutput = false;
  m_inputMaker = NULL;
  m_measureTime = false;
  m_eventMod=1;
	m_writeTextOutput = false;
  m_nminbias = 0;
  m_MB_currentEvent = 0;
  m_ghost_builder= NULL;
}


JetBuilder::~JetBuilder(){
  // First clear JetVars, before we clear the momentmaps :
  m_ntpmaker.clear_vars();
  // Clear algs and their associated JetCollections & JetMomentMaps
  jetalg_map_t::iterator it = m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();
  for(; it!= itE; ++it){
    m_log << DEBUG << "Clearing "<< (*it).first << std::endl;
    algo_data_t algd = (*it).second;
    if ( (algd.alg != NULL) ){ 
      if (algd.internAlg) delete algd.alg;      // don't delete user-given algos
      if(algd.jetlist != NULL ){ // this should not happen
	clear_list(*algd.jetlist);
	delete algd.jetlist; // the JetCollection destructor will clear pointers.	
      }
    }
  }
  m_algmap.clear();
  m_log << DEBUG << "Cleared algmap " << std::endl;
  clear_list(m_inputList);
  m_log << DEBUG << "Cleared input " << std::endl;
  delete m_neg_energyTool;
}


void JetBuilder::configure_input(InputMaker *input, bool saveInput){
  m_inputMaker = input;
  m_savingInput = saveInput;
	m_inputMaker->init();
  
	if(m_savingInput){ 
    std::string name = input->name();
    if (name=="") name = "Input";    
    // Add dummy alg of Input so it is saved in ntuple
    m_algmap[name] = algo_data_t (NULL, &m_inputList, 0);
  }

  m_GeV = input->getGeV();
  m_MeV = input->getMeV();
}

void JetBuilder::configure_output(std::string treename, std::string filename, bool update){
  m_treename = treename;
  m_filename = filename;
  m_updateOutput = update;
}

void JetBuilder::add_text_output(std::string filename) {
  m_writeTextOutput = true;
  m_text_filename = filename;
}

void JetBuilder::add_default_alg(JetTool *jetfinder , bool withIndex ){

  JetAlgorithm *alg;
  std::string name;
  if(! jetfinder) {
    m_log << WARNING << "Jettool is null. Configuring only selectors " << std::endl;
    name = "NoJetFinder";
    alg = new JetAlgorithm(name);

  } else{
    name = jetfinder->name();
    if(m_algmap.find(name) != m_algmap.end() ){
    m_log << ERROR << "JetTool "<<name<< "already configured." << std::endl;
    return;
    }
    m_log << DEBUG << "Configuring default algorithm with finder "<< name << std::endl;
    alg = new JetAlgorithm(name);
		
		// Now add the default tools around the jet finder
		if(m_correct_neg_energy) alg->addTool((JetTool*)m_neg_energyTool); // negative energy handler (if requested)
    alg->addTool(jetfinder);  // The jet finder !! ---------------------------------------
    if(m_correct_neg_energy) alg->addTool((JetTool*)m_neg_energyTool); // negative energy handler (if requested)

    // keep track of this user-added finder
    m_finderList.push_back(jetfinder);
  }
  m_algmap[name] = algo_data_t(alg, new JetCollection(), withIndex);  
}


void JetBuilder::add_custom_alg(JetAlgorithm *jetalg, bool withIndex, bool saveOnlyMomentMap ){
  m_algmap[jetalg->name()] = algo_data_t (jetalg,new JetCollection(), withIndex, saveOnlyMomentMap, false);
}

void JetBuilder::add_eventshape_alg(JetAlgorithm* eventshape_alg){
  m_algmap[eventshape_alg->name()] = algo_data_t (eventshape_alg,new JetCollection(), false, true, false);
}


void JetBuilder::process_events(int nevent, int start){
  
  JetGlobals::globals.inputMaker = m_inputMaker;
  JetGlobals::globals.ntupleMaker = &m_ntpmaker;
  JetGlobals::globals.jetBuilder = this;

  if(m_writeTextOutput) startTextOutput(nevent,start);
  if(!m_inputMaker) {
    m_log << ERROR <<"No input defined! Please use configure_input()." << std::endl;
    return;
  }
  if(m_needInit) init();
  m_inputMaker->moveToEventN(start);
  
	int i =0;
  while( process_next_event()){
    
		write_current_event();
  	m_log << INFO << "****************** End Event " << m_inputMaker->eventNumber() << "************************* " << std::endl;
		if(m_writeTextOutput) continueTextOutput(i);
    i++;
    if(i%m_eventMod==0) std::cout << "Events Processed: " << i << std::endl;
		if (i==nevent) break;
  }
  m_ntpmaker.finalize();
  if(m_writeTextOutput) finishTextOutput();
}
void JetBuilder::write_current_event(){
  // First set Input data if needed:
  std::string inputname = m_inputMaker->name();
  jetalg_map_t::iterator it = m_algmap.find(inputname);
  if(m_savingInput){
    m_ntpmaker.set_data((*it).first, *(*it).second.jetlist );
  }else{
    inputname ="";
  }
  // Now save all other jet collections :
  it = m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();
  for(; it != itE; ++it){
    if(inputname == (*it).first) continue; // (was done above)
    m_ntpmaker.set_data((*it).first, *(*it).second.jetlist );
  }
  m_ntpmaker.fillJets();
}

void JetBuilder::process_one_event(int n){
  if(m_needInit) init();
  m_inputMaker->moveToEventN(n);
  process_next_event();
}


bool JetBuilder::process_next_event(){
  clear_list(m_inputList);

  // get Input
  bool cont = m_inputMaker->fillNextInput(m_inputList);
	
	// skip empty event at end of files for StdTextInput and HepMC input types
   if(!cont && m_inputList.size() == 0) return false;

  m_log << INFO << "****************** Begin Event " << m_inputMaker->eventNumber() << "***********************" << std::endl;
	m_log << INFO << "Input size is: "<< m_inputList.size() << std::endl; 

  // Get min bias if requested
  get_minbias_events();
  if(m_nminbias > 0) m_log << INFO << "Added minbias events. Input size is now: " << m_inputList.size() << std::endl;
  
  // Add ghosts if any
  add_ghosts();
 
	// Run special jet tools that need to modify the global input
	if(m_inputToolList.size()>0)
	{
		m_log << INFO << "Executing input JetTool list" << std::endl;
		jettool_list_t::iterator it = m_inputToolList.begin();	
		jettool_list_t::iterator itE = m_inputToolList.end();	
		m_log << INFO << "Initial input size: " << m_inputList.size() << std::endl;
		for(; it != itE; ++it)
		{
			m_log << INFO << "---- " << (*it)->name() << std::endl;
			(*it)->execute(m_inputList);	
			m_log << INFO << "---- " << "remaining jets: "<< m_inputList.size() << std::endl;
		}
	}

  // Execute all algorithms
  jetalg_map_t::iterator it =  m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();
  for(; it != itE; ++it){
    if((*it).second.alg == NULL) continue;
    m_log << INFO << "==== Algorithm: " << (*it).first << " ====" << std::endl;
    m_log << INFO << "Initial input size: " << m_inputList.size() << std::endl;
    // Get the result list for this alg...
    JetCollection *jetcoll = (*it).second.jetlist;
    // ... clear it...
    clear_list(*jetcoll);
    // ... and fill it
		m_log << INFO << "Executing JetTool list for  "<< (*it).first << std::endl;
    ((*it).second.alg)->execute(m_inputList, *jetcoll);
  }
  return cont;
}

void JetBuilder::add_jetTool(JetTool *jetMoment, std::string name){
  jetalg_map_t::iterator itE = m_algmap.end();
  jetalg_map_t::iterator it = m_algmap.find(name);
  if(it != itE){
    JetAlgorithm * alg = (*it).second.alg;
    alg->addTool(jetMoment);
  }else{
    m_log << ERROR << "No JetAlgorithm '"<<name << "'. Did not insert JetTool "<< jetMoment->name() <<std::endl;
  }
}

void JetBuilder::add_jetTool(JetTool *jetMoment){
  jetalg_map_t::iterator it =  m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();
  for(; it != itE; ++it){
    JetAlgorithm * alg = (*it).second.alg;
    if(alg == NULL) continue;
    alg->addTool(jetMoment);
  }  
}

void JetBuilder::add_jetTool_front(JetTool *jetMoment, std::string name){
  jetalg_map_t::iterator itE = m_algmap.end();
  jetalg_map_t::iterator it = m_algmap.find(name);
  if(it != itE){
    JetAlgorithm * alg = (*it).second.alg;
    alg->addTool_front(jetMoment);
  }else{
    m_log << ERROR << "No JetAlgorithm '"<<name << "'. Did not insert JetTool "<< jetMoment->name() <<std::endl;
  }
}

void JetBuilder::add_jetTool_front(JetTool *jetMoment){
  jetalg_map_t::iterator it =  m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();
  for(; it != itE; ++it){
    JetAlgorithm * alg = (*it).second.alg;
    if(alg == NULL) continue;
    alg->addTool_front(jetMoment);
  }  
}

void JetBuilder::add_jetTool_input(JetTool *tool){
	m_inputToolList.push_back(tool);
}


void JetBuilder::add_moments(JetTool *jetMoment){
  add_jetTool(jetMoment);
}
void JetBuilder::add_moments(JetTool *jetMoment, std::string name){
  add_jetTool(jetMoment, name);
}

JetCollection*  JetBuilder::get_input_jets(){
  return &m_inputList;
}
JetCollection*  JetBuilder::get_jets(std::string jetname){
  jetalg_map_t::iterator it = m_algmap.find(jetname);
  if(it != m_algmap.end() ) return (*it).second.jetlist;
  else return NULL;
}

void JetBuilder::init(){

	// Loop over and init input tool list
	jettool_list_t::iterator toolItr = m_inputToolList.begin();	
	for(; toolItr != m_inputToolList.end();++toolItr)
	{
		m_log << DEBUG << "Initializing input tool: " << (*toolItr)->name() << std::endl;
		(*toolItr)->init(m_inputList.get_JetMomentMap());	
	}
  
	// first register output variables of input collection
  if(m_savingInput){ 
    std::string name = m_inputMaker->name();
    if (name=="") name = "Input";    

		// Schedule MB-overlay-related event moments
  	if(m_nminbias > 0)
		{
			m_inputList.get_JetMomentMap()->schedule_event_moment("nMBevents");
			m_inputList.get_JetMomentMap()->schedule_event_moment("N_MB");
		}
		// allow the input maker to add moments
		m_inputMaker->init_collection(m_inputList);
    m_ntpmaker.addInputJetVar( output_var_style.get_inputjet_var(name, m_inputList.get_JetMomentMap()) );
  }

	// Loop over Algs and register output
  jetalg_map_t::iterator it =  m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();
  for(; it != itE; ++it){
    JetAlgorithm * alg = (*it).second.alg;
    if(alg == NULL) continue;
    if(m_measureTime) alg->doTimeMeasure();

    alg->init();
    std::string name = alg->name();

    // Register this JetAlgorithm in NtupleMaker
    IGenericJetVar * jv = create_jet_var(name, (*it).second );
    m_ntpmaker.addJetVar( jv );
  }  

  m_ntpmaker.init(m_treename, m_filename,m_updateOutput);
  m_needInit = false;
}

IGenericJetVar * JetBuilder::create_jet_var(std::string name, algo_data_t alg_i){

  IGenericJetVar *inputvar = m_ntpmaker.getJetVar(m_inputMaker->name());
  if( !alg_i.saveIndex || !m_savingInput) inputvar = NULL;  // we don't save index of constituents

  JetAlgorithm * alg = alg_i.alg;
  return output_var_style.get_jet_var(name, 
				      alg->get_jet_momentMap(), 
				      inputvar, 
				      alg_i.saveOnlyMomentMap);

}

void JetBuilder::startTextOutput(int nevent,int start) {
  m_textfile << std::left;

  jetalg_map_t::iterator it = m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();

  m_textfile.open(m_text_filename.c_str());
  if(!m_textfile.is_open()) m_log << ERROR << "Output text file \"" << m_text_filename << "\" not created!\n";

  m_textfile << "***********************************************************************" << std::endl;
  m_textfile << "***********************************************************************" << std::endl;
  m_textfile << "***** SpartyJet Text Output File                                  *****" << std::endl;
  m_textfile << "*****-------------------------------------------------------------*****" << std::endl;
  m_textfile << "*****-------------------------------------------------------------*****" << std::endl;
  m_textfile << "*****                   Current Job Description                   *****" << std::endl;
  m_textfile << "*****-------------------------------------------------------------*****" << std::endl;
  m_textfile << "*****   Input File Info:                                          *****" << std::endl;
  m_textfile << "*****     Object Name:  " << std::setw(42) << m_inputMaker->name()  << "*****" << std::endl;
  m_textfile << "*****-------------------------------------------------------------*****" << std::endl;
  m_textfile << "*****   Algorithms:                                               *****" << std::endl;
  for(; it != itE; ++it){
    if(it->first == m_inputMaker->name()) continue;
    m_textfile << "*****      " << std::setw(55) << it->first << "*****" << std::endl;
  }
    m_textfile << "*****-------------------------------------------------------------*****" << std::endl;
  m_textfile << "*****   Orders:                                                   *****" << std::endl;
  m_textfile << "*****     Process " << std::setw(7) << nevent <<  " events, starting at event " 
	     << std::setw(7) << start << "       *****" << std::endl;
  if(m_nminbias > 0) {
    m_textfile << "*****   Add Minimum Bias Events : " << std::setw(10) << m_nminbias << "                      *****" << std::endl;
    if(poisson) m_textfile << "*****     Poisson Option Turned on                              *****" << std::endl;
    m_textfile << "*****     From object name:  " << std::setw(37) << m_MBinput->name()  << "*****" << std::endl;
   }
  m_textfile << "***********************************************************************" << std::endl;
  m_textfile << "***********************************************************************" << std::endl;
}

void JetBuilder::continueTextOutput(int i) {
  jetalg_map_t::iterator it = m_algmap.begin();
  jetalg_map_t::iterator itE = m_algmap.end();

  m_textfile << "\n\n\n***********************************************************************" << std::endl;
  m_textfile << "Event " << i << std::endl;
  for(; it != itE; ++it)
    if(it->first == m_inputMaker->name()) 
      m_textfile << "  " << it->second.jetlist->size() << " four vectors" << std::endl;
  m_textfile << "***********************************************************************" << std::endl;

  it = m_algmap.begin();
  for(; it != itE; ++it){
    if(it->first == m_inputMaker->name()) continue;

    m_textfile << it->first << " Jets\n";
    m_textfile << "           Pt          eta          phi     n      mass\n";
    m_textfile << std::right;
    JetCollection::iterator jit  = it->second.jetlist->begin();
    JetCollection::iterator jitE = it->second.jetlist->end();
    for(; jit != jitE; ++jit) {
      Jet *j = (*jit);
      m_textfile << std::setw(13) << j->pt()
		 << std::setw(13) << j->eta()
		 << std::setw(13) << j->phi()
		 << std::setw(6)  << j->getConstituentNum()
		 << std::setw(13) << j->mass() << std::endl;
    }
    m_textfile << "\n\n";
  }
}

void JetBuilder::finishTextOutput() {
  m_textfile.close();
}

void JetBuilder::add_minbias_events(float n,InputMaker *input,bool on) {
  m_MBinput = input;
	m_MBinput->init();
  m_nminbias = n;
  poisson = on;
  if(poisson)
    m_rand = new TRandom3(abs(time(0)));
  else m_rand = NULL;
}

void JetBuilder::set_minbias_num(float n) {
  m_nminbias = n;
}

  void JetBuilder::set_minbias_poisson(bool on, int seed, int firstMBevt) {
  if(m_rand != NULL) 
    delete m_rand;
  poisson = on;  
  if(seed == 0) seed = abs(time(0));
  if(poisson)
    m_rand = new TRandom3(seed);
  m_MB_currentEvent = firstMBevt;
}



void JetBuilder::get_minbias_events() {
  if(m_nminbias == 0) return;
  bool c;
  m_MBjetList.clear();

  int temp_num;
  
  if(poisson) {
    temp_num = m_rand->Poisson(m_nminbias);
    m_log << INFO << "Poisson Minimum Bias selected " << temp_num << " minbias events." << std::endl;
  }
  else
    temp_num = int(m_nminbias);

	int numInputBefore = m_inputList.size();
  for(int MBi = 0; MBi < temp_num; MBi++)
	{
    m_MBinput->moveToEventN(m_MB_currentEvent);  
    int first_index = m_inputList.size();
    c = m_MBinput->fillNextInput(m_inputList,first_index); // directly fill input list
    if(!c) {               // found no event
      m_MB_currentEvent = 0;  // start back over at beginning
      m_MBinput->moveToEventN(0);
      MBi--;                  // take one away to make up for empty event
    }
    else m_MB_currentEvent++;
  }
	m_inputList.get_JetMomentMap()->set_event_moment("nMBevents",(float)temp_num);
	m_inputList.get_JetMomentMap()->set_event_moment("N_MB",(float)(m_inputList.size()-numInputBefore));
}

  void JetBuilder::adding_active_ghosts(int nghosts){
    if(m_ghost_builder ) {delete m_ghost_builder; m_ghost_builder = NULL;}

    if(nghosts>0){
      m_ghost_builder = new GhostBuilderRandom(nghosts);
    }
  }
  void JetBuilder::reconfigure_ghosts(int nghosts, float rapmin, float rapmax, float phimin, float phimax){
    if( ! m_ghost_builder) adding_active_ghosts(nghosts);
    m_ghost_builder->reconfigure(nghosts, rapmin, rapmax, phimin, phimax);
  }

  void JetBuilder::add_ghosts(){
    if(! m_ghost_builder) return;
    m_ghost_builder->add_ghosts(&m_inputList);
  }  

}  // namespace SpartyJet
