#ifndef _JETBUILDER_HH_
#define _JETBUILDER_HH_

#include <string>
#include <map>
#include <fstream>
#include <TRandom3.h>
#include "CustomMessage.hh"
#include "Jet.hh"
#include "JetCollection.hh"
#include "OutputVarUtils.hh"
#include "NtupleMaker.hh"

class TTree;

namespace SpartyJet { 

class InputMaker;
class IGenericJetVar;
class JetAlgorithm;
class JetTool;
class GhostBuilder;
class JetNegEnergyTool;


class JetBuilder {
public:
  JetBuilder(MessageLevel ml=INFO);
  ~JetBuilder();
  
  /// A pack of infos related to a JetAlgorithm
  class algo_data_t {
  public:
    algo_data_t(){};
    algo_data_t(JetAlgorithm *jalg, JetCollection *list, bool ind, bool onlymap=false, bool intern=true):
      alg(jalg), jetlist(list), saveIndex(ind), saveOnlyMomentMap(onlymap), internAlg(intern) {}
    
    JetAlgorithm *alg;           /// the jet alg
    JetCollection *jetlist;    /// the jet list that will be passed to the jet alg
    bool saveIndex ;             /// save jet constituents indices in final ntuple
    bool saveOnlyMomentMap;      /// if true don't save jets kinematics (used for eventshape algos)
    bool internAlg;              /// Mark this algo as owned by a JetBuilder (otherwise it will not be deleted)
  } ;
  typedef std::map< std::string , algo_data_t> jetalg_map_t;
  typedef std::list<JetTool*> jettool_list_t;
  typedef std::map< std::string,jettool_list_t> jettool_map_t; 

  
  /// Set the input class. if saveInput is true, the input protojet
  /// list will be save in the final ntuple.
  void configure_input(InputMaker *input, bool saveInput = true);

  /// Set the name of the result ntuple (treename) and its file name (filename) by default the 
  /// file is overwritten (update=flase)
  void configure_output(std::string treename, std::string filename, bool update=false);

  /// Build & schedule a JetAlgorithm with the JetTool jetfinder.
  /// This Alg will have a pre- and post-selector plus any moment tool
  /// added to the JetBuilder. If @param withIndex is true, an array relating
  /// input constituent with jets is saved in the output ntuple (the
  /// jetfinder must be compatible with the Jet class constituent)
  /// The jet collection will be named as jetfinder->name()
  void add_default_alg(JetTool *jetfinder , bool withIndex = true);
  
  /// Correction of negative energy of inputs through JetNegEnergyTool 
  void do_correct_neg_energy(bool v){m_correct_neg_energy = v;}

  /// Schedule a full JetAlgorithm. This to allow very specific
  /// configuration for a given algo. Otherwise, use add_default_alg
  /// The jet collection will be named as jetalg->name()
  void add_custom_alg(JetAlgorithm *jetalg, bool withIndex = true, bool saveOnlyMap = false );

  /// Add the JetTool jetmoment to all algorithm already added.
  void add_moments(JetTool *jetMoment);

  /// Add the JetTool jetmoment to algorithm name.
  void add_moments(JetTool *jetMoment, std::string name);

  /// Add a generic tool to all or a given algorithm (currently these are alias of add_moments)
  void add_jetTool(JetTool *jetTool, std::string name);
  void add_jetTool(JetTool *jetTool);
  void add_jetTool_front(JetTool *jetTool,std::string name);
  void add_jetTool_front(JetTool *jetTool);

	void add_jetTool_input(JetTool *jetTool);

  /// Add an eventshape-like algorithm 
  /// (actually configure it not to produce jet vars, only moment map vars)
  void add_eventshape_alg(JetAlgorithm* eventshape_alg);

  /// Add conversion of jet's phi to [0.2Pi]
  void add_phiconversion(){ IGenericJetVar::do_phiconversion=true;};

  /// Add output text file that displays all of the jets for each event
  void add_text_output(std::string filename);
  
  /// Add Minimum-Bias events to data set
  void add_minbias_events(float n,InputMaker *input,bool = false);
  void get_minbias_events();
  void set_minbias_poisson(bool on, int seed=0, int firstMBevt=0);
  void set_minbias_num(float n);


  void adding_active_ghosts(int nghosts);
  void reconfigure_ghosts(int nghosts, float rapmin, float rapmax, float phimin, float phimax);

  void do_time_measure(){m_measureTime=true;};
  
  void process_events(int nevent =-1 , int start =0);
  void process_one_event(int n);
  bool process_next_event();

  // turn off message system
  void silent_mode(){Message::silent_all();}  
	
  // turn off message system
  void set_message_level(MessageLevel l){Message::set_message_level(l);}  
  
  // Set frequency of event number printing
  void print_event_every(int n) { m_eventMod = n;}

  /// Access the input jet collection
  JetCollection* get_input_jets();
  /// Access the jet collection corresponding to jetname
  JetCollection* get_jets(std::string jetname);
  
  double GeV(){ return m_GeV;};
  double MeV(){ return m_MeV;};

  void change_jet_ouptut_var(std::string eta,std::string phi,std::string pt,std::string e,std::string mass, std::string ind, std::string numc ){
    m_ntpmaker.set_var_names(eta,phi,pt,e,mass,ind,numc );
  }



  // Following functions allow finer control of the build by users.

  /// init all JetAlgorithms and register them (or not)
  /// in the NtupleMaker class. Init the NtupleMaker
  void init();

  /// Write all current jet collection into the associated TTree
  void write_current_event();

  void save_ntuple(){m_ntpmaker.finalize();}

  TTree * get_tree(){return m_ntpmaker.get_tree();}


  /// Holds configuration of output variable 
  OutputVarFactory output_var_style;

protected:

  double m_GeV;
  double m_MeV;

  
  // make text output file
  void startTextOutput(int nevent,int start);
  void continueTextOutput(int i);
  void finishTextOutput();


  // build jet var
  IGenericJetVar * create_jet_var(std::string name, algo_data_t alg_i);

  bool m_needInit;
  bool m_savingInput;

  void add_ghosts();
  GhostBuilder *m_ghost_builder;

  // Manage input particle with negative energy (calorimeter clusters....)
  bool m_correct_neg_energy;
  JetNegEnergyTool *m_neg_energyTool;
  
  jetalg_map_t m_algmap;       // stores Full Jet Algs
  jettool_list_t m_finderList; // stores finder Tool
  jettool_list_t m_inputToolList; // stores input Tools

  JetCollection m_inputList; /// input jets

  NtupleMaker m_ntpmaker;      /// manage final ntuple output
  bool m_updateOutput;
  InputMaker *m_inputMaker;    /// manage input, given by user

  std::string m_treename;
  std::string m_filename;

  // for text output
  std::string m_text_filename;
  bool m_writeTextOutput;
  ofstream m_textfile;
 
  // for minbias events
  float m_nminbias;
  InputMaker *m_MBinput;
  int m_MB_currentEvent;
  Jet::jet_list_t m_MBjetList;  /// minbias jets
  bool poisson;
  TRandom3* m_rand;

  bool m_measureTime;

  Message m_log;
  int m_eventMod;
};


}  // namespace SpartyJet
#endif
