#ifndef _JETHISTORY_HH_
#define _JETHISTORY_HH_

#include <iostream>
#include "JetCore/Jet.hh"

/// Here is coded a history structure for jets.
/// There are 2 main goals behind this design
///  - code the structure and its feature only once and independently of any other class
///  - being able to provide this structure to any other class.
/// First part is coded in HistoryNode, the second in HistoryProvider.
/// To give history structure to class A make it inherit HistoryProvider<A>.
/// For now, the only thing needed in A is that it has the instruction 
/// history.item = this; 
/// before any history feature is used.

namespace SpartyJet { 


  typedef std::vector<Jet*> hconstit_list_t;
  /// \class HistoryNode<T> defines a history structure
  /// 
  ///  HistoryNode holds a a pointer to a arbitrary class T which needs history.
  ///  It expects the class T has a public function hindex() (returning an index in a history sequence)  
  template<class T>
  class HistoryNode  {
  public:
    typedef  std::vector< T* > hist_list_t;


    class history_index_sort {
    public:
      bool operator()(const T *o1, const T* o2){
        //return o1->hnode.hindex < o2->hnode.hindex;
        return o1->hindex() < o2->hindex();
      }
    };


    HistoryNode():parent1(0),parent2(0),child(0), item(0),dij(-1), jet(0), hindex(-1), m_ownjet(false){}

    virtual ~HistoryNode(){ if(m_ownjet) delete jet;}


    /// a safe way to access the jet    
    Jet* getJet() {return buildJet();}


    void retrieveConstitutentJets(hconstit_list_t & constits);

    /// retrieve all parents of this node, putting them in nodes. Bool arguments control which node are added.
    void retrieveParentNodes( hist_list_t &nodes , bool addOrphan, bool addNonOrphan );
    /// shortcut for retrieveParentNodes(addOrphan=false,addNonOrphan=true)
    hist_list_t fullHistory();    


    HistoryNode *parent1, *parent2;
    HistoryNode *child;
    T* item; 

    float dij;
    Jet* jet;
    int hindex;



  protected:
    bool m_ownjet;

    
    Jet* buildJet(){
      
      if(jet) {
        //std::cout << " jet already exists "<< std::endl;
        return jet;
      }
      hconstit_list_t  constits;
      retrieveConstitutentJets(constits);
      if ( constits.empty()) {
        std::cout << " empty constituents hindex="<< hindex << std::endl;
      }
      hconstit_list_t::iterator it = constits.begin();
      hconstit_list_t::iterator itE = constits.end();
      jet = new Jet();
      for( ; it !=itE; ++it){
        jet->addConstituent(*it);
      }
      m_ownjet = true;
      return jet;
    }

  };

  template<class T>
  void HistoryNode<T>::retrieveConstitutentJets(hconstit_list_t & constits){
    if(parent1) parent1->retrieveConstitutentJets(constits);
    else {
      if( !jet) std::cout << this <<"   !!! PBM no parent1 & no jet at "<<hindex << std::endl;
      constits.push_back(jet);
      return;
      }
    if(parent2) parent2->retrieveConstitutentJets(constits);
  }


  template<class T>
  typename HistoryNode<T>::hist_list_t  
  HistoryNode<T>::fullHistory(){
    hist_list_t hist;
    retrieveParentNodes( hist, false, true);
    std::sort(hist.begin() , hist.end() , history_index_sort() );
    return hist;
  }

  template<class T>
  void HistoryNode<T>::retrieveParentNodes( std::vector<T*> &nodes , bool addOrphan, bool addNonOrphan ){
    if(parent2) parent2->retrieveParentNodes(nodes, addOrphan, addNonOrphan);
    if(parent1) parent1->retrieveParentNodes(nodes, addOrphan, addNonOrphan);
    else {
      //if( !jet) std::cout << " !!! PBM no parent1 & no jet at "<<hindex << std::endl;
      if(addOrphan) nodes.push_back(item);
      return;
    }
    if(addNonOrphan) nodes.push_back(item);
  }



  /// \class HistoryProvider
  ///
  /// This class works together with HistoryNode to provide a history structure to a class
  /// The intended use of it is 'class A : public HistoryProvider<A> '.
  ///
  /// The advantage of this mechanism is A will inherit entirely the history structure 
  /// from HistoryNode without having any code to implement on its side a part from 
  ///     HistoryProvider<A>::init( something ); // optionnal
  ///     History.item = this;                   // Required !!!
  /// which needs to be executed before using hstory features.

  template<class T>
  class HistoryProvider {
  public:
    typedef typename HistoryNode<T>::hist_list_t hist_list_t ;

    
    HistoryProvider(){init();}
    HistoryProvider(const HistoryProvider &h){init(h);} // parent/child not copied !!
    HistoryProvider & operator=(const HistoryProvider &h){init(h); return *this;} // parent/child not copied !!

    void init(){history.item = static_cast<T*>(this);}

    void retrieveConstitutentJets(hconstit_list_t & constits){history.retrieveConstitutentJets(constits);};
    

    
    void retrieveParentNodes( hist_list_t &nodes , bool addOrphan, bool addNonOrphan ){history.retrieveParentNodes(nodes,addOrphan,addNonOrphan);}
    hist_list_t fullHistory() {return history.fullHistory();}


    T * parent1(){return history.parent1 ? history.parent1->item : NULL ;}
    T * parent2(){return history.parent2 ? history.parent2->item : NULL ;}
    T * child(){return history.child ? history.child->item : NULL ;}

    void setParent1(T* p){history.parent1 = & (p->history);}
    void setParent2(T* p){history.parent2 = & (p->history);}
    void setChild(T* p){history.child     = & (p->history);}

    
 
    /// safe access to jet (including a rebuild if necessary)
    Jet* getJet(){return history.getJet();}
    /// direct access to jet 
    Jet* jet() const {return history.jet;}

    float dij() const {return history.dij;}
    int hindex()const {return history.hindex;}


    void setJet(Jet* j){history.jet = j;}
    void setDij(float dij){history.dij = dij;}
    void setHindex(int hindex){history.hindex = hindex;}

    
    template<class HISTORY_t>
      void init(const HISTORY_t &hnode ){
      history.dij = hnode.dij();    
      history.hindex = hnode.hindex();    
      history.jet = hnode.jet();
      init();
    }

  protected:
    HistoryNode<T> history;
  };





  /// A super simple actual use of history node :
  class HistoryElement : public HistoryProvider<HistoryElement>{
  public:
    //HistoryElement(){};
    virtual ~HistoryElement(){};
  };




}
#endif
