#ifndef _JETMOMENT_MAP_HH_
#define _JETMOMENT_MAP_HH_

#include <map>
#include <vector>
#include <string>

#include "Jet.hh"
/////////////////////////////////////////////////////////////////////////////
/// \class JetMomentMap  
/// 
/// \brief associate additional data to jets
/// 
/// A JetMomentMap is meant to be associated to a JetAlgorithm, and is used
/// to relates any additional quantities ("moments") to the jets produced by the JetAlgorithm.
/// It also allows to keep additional quantities associated to the algorithm itself 
///
/// The implementation is ugly : only float or vector of float are allowed as jet moment.
/// this is because : 
///   - it is difficult to have generic way of writing data into ROOT
///   - we want to keep ROOT ntuple as simple as possible
///   - is it worth to do complicated stuff with heavily templated classes ?
namespace SpartyJet { 
/////////////////////////////////////////////////////////////////////////////////
class JetMomentMap {
public:
  typedef std::vector<std::string> name_store_t;
  typedef std::vector<float> value_store_t;
  typedef std::vector<std::vector<float> > valuearray_store_t;
  
  typedef std::map<Jet*, value_store_t*> moment_map_t;
  typedef std::map<Jet*, valuearray_store_t*> momentarray_map_t;

  ~JetMomentMap(){clear();}

  // add/get JetMoments 
  void schedule_jet_moment(std::string name);
  void unschedule_jet_moments();//{m_momnames.clear();}
  void remove_jet_moment(std::string mom_name);
  void set_jet_moment(int pos, Jet* jet, float value);
  value_store_t* get_jet_moments(Jet* jet) const;
  int  get_jet_momentPos(std::string name) const;
  int num_jet_moment() const {return m_momnames.size();}
  std::string moment_name(int i) const {return m_momnames[i];}
  float get_jet_moment(std::string mom_name,  Jet* jet) const;
  float get_jet_moment(int i,  Jet* jet) const;


  // add/get JetMoments arrays
  void schedule_jet_moment_array(std::string name);
  void unschedule_jet_moment_arrays() {m_momarray_names.clear();}
  inline void set_jet_moment_array(int moment_pos, Jet* jet, value_store_t & theArray);
  inline valuearray_store_t* get_jet_moment_arrays(Jet* jet) const;
  int  get_jet_moment_arrayPos(std::string name) const;
  int num_jet_moment_array() const {return m_momarray_names.size();}
  std::string moment_array_name(int i) const {return m_momarray_names[i];}
  std::vector<float> get_jet_moment_array(std::string mom_name,  Jet* jet) const;
  std::vector<float> get_jet_moment_array(int i,  Jet* jet) const;

  // add/get global values
  void schedule_event_moment(std::string name);
  inline void set_event_moment(std::string name, float value);
  inline void set_event_moment(int pos, float value);
  inline value_store_t* get_event_moment() const;
  int num_event_moment() const {return m_globalnames.size();}
  std::string event_moment_name(int i) const {return m_globalnames[i];}
  int  get_event_momentPos(std::string name) const;

  // add/get global array of values
  void schedule_event_moment_array(std::string name);
  inline void set_event_moment_array(std::string name, value_store_t &theArray);
  inline void set_event_moment_array(int pos, value_store_t &theArray);
  inline valuearray_store_t* get_event_moment_array() const;
  int num_event_moment_array() const {return m_globalarraynames.size();}
  std::string event_moment_array_name(int i) const {return m_globalarraynames[i];}
  int  get_event_moment_arrayPos(std::string name) const;



  //void init();
  void clear();


  void swap(JetMomentMap &map);

  void copy_structure(JetMomentMap &map);

protected:
  
  moment_map_t m_map;
  name_store_t m_momnames;
  momentarray_map_t m_array_map;
  name_store_t m_momarray_names;
  
  name_store_t m_globalnames;
  name_store_t m_globalarraynames;
  

  void i_set_jet_moment(int pos, Jet* jet, float value);
  value_store_t* i_get_jet_moments(Jet* jet) const;

  void i_set_jet_moment_array(int pos, Jet* jet,  value_store_t & theArray);
  valuearray_store_t* i_get_jet_moment_arrays(Jet* jet)const;

};









#ifndef G__DICTIONARY
// prevent dict generation by rootcint for this part of header
inline void JetMomentMap::set_jet_moment(int pos, Jet* jet, float value){
  if (jet ==NULL) return; // issue WARNING ?
  i_set_jet_moment(pos,jet,value);
}
inline JetMomentMap::value_store_t* JetMomentMap::get_jet_moments(Jet* jet) const {
  if (jet ==NULL) return NULL; // issue WARNING ?
  return i_get_jet_moments(jet);
}

inline void JetMomentMap::set_jet_moment_array(int pos, Jet* jet, value_store_t & theArray){
  if (jet ==NULL) return; // issue WARNING ?
  i_set_jet_moment_array(pos,jet,theArray);
}
inline JetMomentMap::valuearray_store_t* JetMomentMap::get_jet_moment_arrays(Jet* jet) const {
  if (jet ==NULL) return NULL; // issue WARNING ?
  return i_get_jet_moment_arrays(jet);
}

  inline void JetMomentMap::set_event_moment(std::string name, float value){
  int pos = get_event_momentPos( name);
  i_set_jet_moment(pos,NULL,value);
}

  inline void JetMomentMap::set_event_moment(int pos, float value){
  i_set_jet_moment(pos,NULL,value);
}

inline JetMomentMap::value_store_t* JetMomentMap::get_event_moment() const {
  return i_get_jet_moments(NULL);
}

inline void JetMomentMap::set_event_moment_array(std::string name, value_store_t & theArray){
  int pos = get_event_moment_arrayPos( name);
  i_set_jet_moment_array(pos,NULL,theArray);
}

inline void JetMomentMap::set_event_moment_array(int pos, value_store_t & theArray){
  i_set_jet_moment_array(pos,NULL,theArray);
}
inline JetMomentMap::valuearray_store_t* JetMomentMap::get_event_moment_array() const {
  return i_get_jet_moment_arrays(NULL);
}

#endif
}  // namespace SpartyJet
#endif
