#ifndef SJ_TREEBRANCHUTILS_h
#define SJ_TREEBRANCHUTILS_h



#include "TTree.h"
#include "TString.h"
#include <vector>
#include <iostream>

#include "Jet.hh"
#include "InputMaker.hh"
/////////////////////////////////////////////////////////
/// \class BranchWrap
///
/// This class abstract the TBranch in order to provide a 
/// unique interface to TBranch::SetBranchAddress (set-up for reading) and 
/// TBranch::Branch (set-up for writing)
/// The derived classes of BranchWrap implement the concrete
/// type of the TTree variable, holding an object of this type
/// ex : vector<double> or float or ...
/////////////////////////////////////////////////////////


namespace SpartyJet {
  class BranchWrap {
  public:
    BranchWrap(TString n="") :  m_b(0), m_name(n)  {};
    virtual ~BranchWrap(){};
    
    virtual void SetBranchAddress(TTree * t){}
    virtual void registerBranch(TTree *t){}
    virtual void resize(int i){} // useful only for derived class using vectors
    

    int GetEntry(int i){
      //std::cout << "GetEntry  "<< m_name << "  "<< m_b->GetName() << std::endl;
      if(m_b) { return m_b->GetEntry(i); }
      else return 0;}
    void set_name(TString n){m_name=n;}
    void set_name(std::string n){m_name=n.c_str();}
    TString get_name(){return m_name;}
  
    TBranch *m_b;  
    
    TString m_name;
  
  };

  class BranchWrap1D : public BranchWrap {
  public:
    BranchWrap1D(TString n = "") : BranchWrap(n), m_refN(0) {}
    virtual ~BranchWrap1D(){}
    
    TString get_refN_name(){
      if(m_refN) return m_refN->get_name();
      else return "";
    }

    void set_refN(BranchWrap *b){m_refN = b;}

    BranchWrap * m_refN;
  };

  template<class T>
  class BranchWrap_vector : public BranchWrap1D {
  public:
    BranchWrap_vector(TString n = "") : BranchWrap1D(n), m_vec(0) {}
    virtual ~BranchWrap_vector(){ if (m_vec) {m_vec->clear();  delete m_vec;} }

  T& operator[](int i){return (*m_vec)[i];}
  
  int size(){return m_vec->size();}
  int size_max(){return 999999999;}

  
  virtual void SetBranchAddress(TTree * t){
    m_b = t->GetBranch(m_name);
    if(m_b){
      t->SetBranchAddress(m_name , &m_vec);
      //std::cout << " Setting TBranch Address vector : "<< m_vec  << std::endl;
    }
    else
      std::cout << " WARNING !! no TBranch named : "<< m_name  << std::endl;
  }


    virtual void registerBranch(TTree *t){
      m_b = t->Branch(m_name, &m_vec);
    }

    virtual void resize(int i){m_vec->resize(i);}

  std::vector<T> *m_vec;  

};




  template<class T>
  TString get_type_string(){return "";}
  template<class T>
  TString get_type_stringLong(){return "";}


#ifdef OUTPUTVARUTILS_CC 
// so that these functions are defined only once
  template<>
  TString get_type_string<float>(){return "F";};
  template<>
  TString get_type_string<int>(){return "I";};
  template<>
  TString get_type_string<double>(){return "D";};
  template<>
  TString get_type_stringLong<float>(){return "float";};
  template<>
  TString get_type_stringLong<int>(){return "int";};
  template<>
  TString get_type_stringLong<double>(){return "double";};
#endif



  template<class T, int N_T=100>
  class BranchWrap_array : public BranchWrap1D {
  public:
    
    BranchWrap_array(TString name="") : BranchWrap1D(name) {}
    virtual ~BranchWrap_array(){};

    T& operator[](int i){return m_vec[i];}
    
    int size(){return N_T;}
    int size_max(){return N_T;}
  
    virtual void SetBranchAddress(TTree * t){
      t->SetBranchAddress(m_name , m_vec); 
      m_b = t->GetBranch(m_name);
      if(!m_b)      std::cout << " WARNING !! no TBranch named : "<< m_name  << std::endl;
    }

    virtual void registerBranch(TTree *t){
      // build the type as in e.g. "myvar[150]/F"
      TString refname = get_refN_name();
      if (refname=="") refname += N_T;
      TString type = m_name; type +="[";
      type+= refname; 
      //type+= "20000"; testing
      type+="]/";
      type+= get_type_string<T>();
      m_b = t->Branch(m_name, m_vec,type);
      //std::cout <<" ---- registering "<< m_name << "   type= "<< type <<"  max size="<< N_T <<"  at "<< m_b<< std::endl;
    }

    T m_vec[N_T];  
    
  };


  template<class T>
  class BranchWrap_simple : public BranchWrap {
  public:
    BranchWrap_simple(TString n="") : BranchWrap(n) {}
    virtual ~BranchWrap_simple(){}

    void operator=(T v){m_val=v;}
    operator T& (void){return m_val;}
    
    virtual   void SetBranchAddress(TTree * t ){
      t->SetBranchAddress(m_name , &m_val, &m_b); 
      if(!m_b)      std::cout << " WARNING !! no TBranch named : "<< m_name  << std::endl;
    }

    virtual void registerBranch(TTree *t){
      // build the type as in e.g. "myvar[150]/F"
      TString type = m_name ;
      type+="/";
      type+= get_type_string<T>();
      //std::cout <<" ---- registering "<< m_name << "   type= "<< type << std::endl;
      m_b = t->Branch(m_name, &m_val,type);
    }

    T m_val;  
    
  };


  template<class T>
  class BranchWrap_vector2D : public BranchWrap1D {
  public:
    BranchWrap_vector2D(TString n = "") : BranchWrap1D(n), m_vec(0) {}
    virtual ~BranchWrap_vector2D(){ if (m_vec) {m_vec->clear();  delete m_vec;} }  

    std::vector<T>& operator[](int i){return (*m_vec)[i];}
  
    int size(){return m_vec->size();}
  
		void set(int i, int j, T v) { m_vec->at(i).push_back(v); }
		
    virtual void SetBranchAddress(TTree * t){
      // no read for now
    }
    
    virtual void registerBranch(TTree *t){
			TString type = "vector<vector<";
			type += get_type_stringLong<T>();
			type += ">>";
      t->Branch(m_name,type,&m_vec);
    }

    virtual void resize(int i){m_vec->resize(i);};
    
    std::vector<std::vector<T> > *m_vec;  
    
};


  template<class T, int N1 = 100, int N2=10>
  class BranchWrap_array2D : public BranchWrap1D {
  public:
    
    BranchWrap_array2D(TString name="") : BranchWrap1D(name) {}
    virtual ~BranchWrap_array2D(){}
    
    T* operator[](int i){return m_vec[i];}
    
    int size(){return N1*N2;}
  
		void set(int i, int j, T v) { m_vec[i][j]= v; }

    virtual void SetBranchAddress(TTree * t){
      // No read for now.
    }

    virtual void registerBranch(TTree *t){
      // build the type as in e.g. "myvar[150]/F"
      TString refname = get_refN_name();
      if (refname=="") refname += N1;
      TString type = m_name +"[";
      type+= refname; type+="][";
      type+= N2; type+="]/";
      type+= get_type_string<T>();
      t->Branch(m_name, m_vec,type);
    }

    T m_vec[N1][N2];  
    
  };


  // *********************************************************
  // *********************************************************
  // Collections


  typedef std::list<BranchWrap*> branchlist_base_t;

  ////////////////////////////////////////////////////////////////
  /// \class IFourVecCollection 
  /// \brief an interface to represent a collection of 4vec stored as flat variables in a TTree
  ///
  /// An interface to represent a collection of 4vec stored as flat variables in a TTree
  /// It provide functions to set the name of the TTree variables
  /// To associate variable addresses with TBranches (SetBranchAddress)
  /// to read the the collection from TTree (fill_collection)
  ////////////////////////////////////////////////////////////////
  class IFourVecCollection {
  public:
		virtual ~IFourVecCollection() {};
    virtual void set_prefix(TString p) = 0;
    virtual void set_suffix(TString s) = 0;
    virtual void set_n_name(TString s) = 0;
    virtual void set_var_names(TString v1,TString v2,TString v3,TString v4) = 0;
    virtual void add_var(TString name) = 0;
    virtual void set_massless(bool m) = 0;
    virtual void set_reject_bad_input(bool m) = 0;
    virtual void SetBranchAddress(TTree * t ) =0 ;
    virtual void GetEntry(int i) = 0;
    
    virtual void fill_collection(Jet::jet_list_t &inputList, int start_index = 0) = 0;
    virtual void fill_collection(int eventn, Jet::jet_list_t &inputList, int start_index = 0) = 0;

  };

  ////////////////////////////////////////////////////////////////
  /// \class FourVecCollection 
  /// \brief A templated generic implementation of IFourVecCollection
  ///
  /// A templated generic implementation of IFourVecCollection.
  /// The template parameter BType is expected to be BranchWrap_array 
  ///  or BranchWrap_vector
  ////////////////////////////////////////////////////////////////
  template<class BType>
  class FourVecCollection : public branchlist_base_t, public IFourVecCollection{
  public:
    FourVecCollection(TString prefix, TString suffix="") : branchlist_base_t() ,m_prefix(prefix), m_suffix(suffix) {
      push_back(&m_n);
      m_n.m_name=prefix+"N"+suffix;
      push_back(&m_v1);push_back(&m_v2);push_back(&m_v3);push_back(&m_v4);
      m_masslessMode = false;
      m_reject_bad_input = true;
    };
    virtual ~FourVecCollection(){};
  
    void set_prefix(TString p){m_prefix = p;}
    void set_suffix(TString s){m_suffix = s;}
    void set_n_name(TString s){m_n.m_name = m_prefix+s+m_suffix;}
    void set_var_names(TString v1,TString v2,TString v3,TString v4){
      m_v1.m_name = m_prefix+v1+m_suffix;
      m_v2.m_name = m_prefix+v2+m_suffix;
      m_v3.m_name = m_prefix+v3+m_suffix;
      m_v4.m_name = m_prefix+v4+m_suffix;
    }

    void add_var(TString name){ push_back( (BranchWrap*) new BType(name));}
    void add_var(BranchWrap * bw){ push_back( bw );}

    virtual void set_massless(bool m){m_masslessMode = m;}    
    virtual void set_reject_bad_input(bool m) {m_reject_bad_input = m;}

    virtual   void SetBranchAddress(TTree * t ){
      branchlist_base_t::iterator it = begin();
      for(; it != end(); ++it) (*it)->SetBranchAddress(t);
    }

    virtual   void GetEntry(int i){
      branchlist_base_t::iterator it = begin();
      for(; it != end(); ++it) (*it)->GetEntry(i);
    }

    
    virtual void fill_collection(Jet::jet_list_t &inputList, int start_index = 0) {};
    virtual void fill_collection(int eventn, Jet::jet_list_t &inputList, int start_index = 0) {
      this->GetEntry(eventn);
      this->fill_collection(inputList, start_index);
    };

    TString m_prefix;
    TString m_suffix;
    BranchWrap_simple<int> m_n;
    
    bool m_masslessMode;
    bool m_reject_bad_input;
    BType m_v1;
    BType m_v2;
    BType m_v3;
    BType m_v4;

  };



  // *********************************************************
  ////////////////////////////////////////////////////////////////
  /// \class EtaPhiPtECollection
  /// \brief FourVecCollection where the variables are eta,phi,pt
  ///
  /// FourVecCollection where the variables are eta,phi,pt
  ////////////////////////////////////////////////////////////////

  template<class BType>
  class EtaPhiPtECollection : public FourVecCollection<BType>  {
  public:
    EtaPhiPtECollection(TString base,TString suffix="") : FourVecCollection<BType>(base,suffix) {
      this->set_var_names("Eta","Phi","Pt","E");
    };
    

    virtual void fill_collection( Jet::jet_list_t &inputList,  int start_index =0) {
      BType & eta = this->m_v1;      BType & phi = this->m_v2;      BType & pt = this->m_v3;      BType & e = this->m_v4;
      int index= start_index;
      for(int i=0; i< this->m_n; i++){
	double pti;
	if( this->m_masslessMode) pti = e[i]/cosh(eta[i]);
	else pti = pt[i];
	double px = pti* cos(phi[i]);
	double py = pti* sin(phi[i]);
	double pz = pti*sinh(eta[i]);
	
	//log << " jet "<< i << std::endl;
	Jet* j = new Jet(px,py,pz,e[i],index);  
	if(!this->m_reject_bad_input || InputMaker::check_4vector(j) ) {
	  inputList.push_back(j);
	  index++;
	} else delete j;
      }
    }      
  };

  // *********************************************************
  ////////////////////////////////////////////////////////////////
  /// \class PxPyPzECollection
  /// \brief FourVecCollection where the variables are px,pypz,e
  ///
  /// FourVecCollection where the variables are px,pypz,e
  ////////////////////////////////////////////////////////////////

  template<class BType>
  class PxPyPzECollection : public FourVecCollection<BType>  {
  public:
    PxPyPzECollection(TString base,TString suffix="") : FourVecCollection<BType>(base,suffix) {
      this->set_var_names("Px","Py","Pz","E");
    };

    virtual void fill_collection( Jet::jet_list_t &inputList, int start_index =0 ) {
      BType & px = this->m_v1;      BType & py = this->m_v2;      BType & pz = this->m_v3;      BType & e = this->m_v4;
      int index= start_index;
      for(int i=0; i< this->m_n; i++){
	Jet* j = new Jet(px[i],py[i],pz[i],e[i],index);  
	if(!this->m_reject_bad_input || InputMaker::check_4vector(j) ) {
	  inputList.push_back(j);
	  index++;
	} else delete j;
      }
    }      
  };
  // *********************************************************
    
  ////////////////////////////////////////////////////////////////
  /// \class EtaPhiPtMCollection
  /// \brief FourVecCollection where the variables are px,pypz,e
  ///
  /// FourVecCollection where the variables are px,pypz,e
  ////////////////////////////////////////////////////////////////
  template<class BType>
  class EtaPhiPtMCollection : public FourVecCollection<BType>  {
  public:
    EtaPhiPtMCollection(TString base,TString suffix="") : FourVecCollection<BType>(base,suffix) {
      this->set_var_names("Eta","Phi","M","E");
    };


    virtual void fill_collection( Jet::jet_list_t &inputList, int start_index = 0) {
      BType & eta = this->m_v1;      BType & phi = this->m_v2;      BType & pt = this->m_v3;      BType & m = this->m_v4;
      int index=start_index;
      for(int i=0; i< this->m_n; i++){
	
	if(fabs( eta[i] ) >500 ) continue; // this will produce nan

	double px = pt[i]* cos(phi[i]);
	double py = pt[i]* sin(phi[i]);
    
	double pz = pt[i]*sinh(eta[i]);
	double p  = pt[i]*cosh(eta[i]);
	
	double e  = sqrt(p*p + m[i]*m[i]);
	
	
	Jet* j = new Jet(px,py,pz,e,index);  
	if(!this->m_reject_bad_input || InputMaker::check_4vector(j) ) {
	  inputList.push_back(j);
	  index++;
	} else delete j;
      } 
            
    }      
  };



}




#endif
