#include "CalorimeterSimTool.hh"

#include "JetCore/Jet.hh"
#include "JetCore/CommonUtils.hh"

namespace SpartyJet { 

CalorimeterSimTool::CalorimeterSimTool(float etaLimit,float deltaEta,float deltaPhi,bool set2Pi): 
	JetTool("CalorimeterSimTool"),
	m_minEta(-1*etaLimit),
	m_maxEta(   etaLimit),
	m_useZeroTo2Pi(set2Pi)
	{
		SetTowerSize(deltaEta,deltaPhi);
		m_nEta = (int)(2*etaLimit/m_deltaEta);
	}


void CalorimeterSimTool::init(JetMomentMap *mmap) 
{
	// Create Tower Grid
	m_log << WARNING << "Creating tower grid of: " 
				<< m_nEta << " bins in eta of size " << m_deltaEta << " and: " 
				<< m_nPhi << " bins in phi of size " << m_deltaPhi << std::endl;
	m_towerEnergy = std::vector<std::vector<double> >(m_nEta,std::vector<double>(m_nPhi,0.) );
	m_towerNparticles = std::vector<std::vector<int> >(m_nEta,std::vector<int>(m_nPhi,0) );
}
void CalorimeterSimTool::execute(JetCollection &theJets) 
{
	// Loop on all jets and insert into calorimeter grid
	JetCollection::iterator jit = theJets.begin(); 
	JetCollection::iterator jitE = theJets.end(); 
	int etaBin,phiBin;
	for(; jit!=jitE;++jit)
	{
		GetEtaPhiBins(*jit,etaBin,phiBin);
		if(etaBin > -1)
		{
			m_towerEnergy[etaBin][phiBin] += (*jit)->e();
			m_towerNparticles[etaBin][phiBin]++;
		}
	}

	jit = theJets.begin();
	// Loop on all towers, if particle was inserted, store in inputJets
	for(int eBin=0;eBin<m_nEta;eBin++)
	{
		for(int pBin=0;pBin<m_nPhi;pBin++)
		{
			if(m_towerNparticles[eBin][pBin] > 0 )
			{
				Double_t eta = m_minEta+(eBin+0.5)*m_deltaEta;
				Double_t phi = (pBin+0.5)*m_deltaPhi;
				if(!m_useZeroTo2Pi) phi = to_minusPI_PI(phi);
				double energy = m_towerEnergy[eBin][pBin] ;
				double pt = energy/cosh(eta);
				double px = pt*cos(phi);
				double py = pt*sin(phi);
				double pz = pt*sinh(eta);
				(*jit)->SetPxPyPzE(px,py,pz,energy);
				//std::cout << "Built Calorimeter Tower "<< (*jit)->index() <<  " of: " << "eta = " << eta << " phi = " << phi << " Energy = " << energy << std::endl;
				++jit;
			}
		}
	}
	// Erase any input jets not overwritten by tower
	theJets.erase(jit,jitE);

	// Reset tower grid
	for(int i=0;i<m_nEta;i++)
	{
		std::fill(m_towerEnergy[i].begin(),m_towerEnergy[i].end(),0.);
		std::fill(m_towerNparticles[i].begin(),m_towerNparticles[i].end(),0);
	}
}

void CalorimeterSimTool::GetEtaPhiBins(Jet *jet,int &etaBin, int&phiBin)
{
	if((jet->eta() < m_minEta) || (jet->eta() > m_maxEta))
	{
		etaBin = -1;
		phiBin = -1;
	}
	else
	{
		float phi = to_zero_2PI(jet->phi());
		double absEta = jet->eta() - m_minEta;
		etaBin = (int)(absEta/m_deltaEta);
		phiBin = (int)(phi/m_deltaPhi);

	}

}


} // end SpartyJet namespace
