// emacs, this is -*- C++ -*-
#ifndef FASTKTJET_KTPARAMS_H
#define FASTKTJET_KTPARAMS_H

#ifndef STD_STRING_H
#include <string>
#define STD_STRING_H
#endif
#include <list>

#include "JetCore/Jet.hh"

namespace SpartyJet { 
typedef Jet::jet_list_t jetcollection_t;


namespace FastKtJet {
  class KtJetInfo;
  typedef std::list<FastKtJet::KtJetInfo*> KtJetInfo_list;
  class KtLists; 
  // *****************************************************************
  // Distances 
  // *****************************************************************
  class KtDistance {
  public:

    enum KtDistanceType {
      DeltaR = 1,
      Angular ,
      Reversed
    };
    /** virtual destructor needed */
    virtual ~KtDistance() {}
    /** Jet Kt */
    virtual float operator()(const KtJetInfo *) const = 0;
    /** Pair Kt */
    virtual float operator()(const KtJetInfo *, const KtJetInfo *) const = 0;
    /** Pseudo-geometrical dist Kt */
    virtual float geodist(const KtJetInfo *, const KtJetInfo *) const = 0;
    /** Name of scheme */
    virtual std::string name() const = 0;
  };
  

  KtDistance* getDistanceScheme(int dist);

  class KtDistanceDeltaR : public KtDistance {
  public:
    KtDistanceDeltaR();
    virtual ~KtDistanceDeltaR(){};
    /** Jet Kt */
    float operator()(const KtJetInfo*) const;
  /** Pair Kt */
    float operator()(const KtJetInfo*, const KtJetInfo*) const;
    /** Pseudo-geometrical dist Kt */
    inline float geodist(const KtJetInfo *, const KtJetInfo *) const ;
    /** Name of scheme */
    std::string name() const;
  private:
    std::string m_name;
  };


  class KtDistanceAngular : public KtDistance {
  public:
    KtDistanceAngular();
    virtual ~KtDistanceAngular(){};
    /** Jet Kt */
    float operator()(const KtJetInfo*) const;
  /** Pair Kt */
    float operator()(const KtJetInfo*, const KtJetInfo*) const;
    /** Pseudo-geometrical dist Kt */
    inline float geodist(const KtJetInfo *, const KtJetInfo *) const ;
    /** Name of scheme */
    std::string name() const;
  private:
    std::string m_name;
  };
  

  class KtDistanceReversed : public KtDistance {
  public:
    KtDistanceReversed();
    virtual ~KtDistanceReversed(){};
    /** Jet Kt */
    float operator()(const KtJetInfo*) const;
  /** Pair Kt */
    float operator()(const KtJetInfo*, const KtJetInfo*) const;
    /** Pseudo-geometrical dist Kt */
    inline float geodist(const KtJetInfo *, const KtJetInfo *) const ;
    /** Name of scheme */
    std::string name() const;
  private:
    std::string m_name;
  };
  
  // *****************************************************************  
  // Recombination schemes  
  // *****************************************************************

  class KtRecom {
  public:  
    enum KtRecomType {
      E = 1,
      Pt
    };

    /** virtual destructor needed */
    virtual ~KtRecom() {}
    /** Return merged 4-momentum */
    virtual void combine( KtJetInfo * jet,  KtJetInfo * jet_toadd)  = 0;
    /** Name of scheme */
    virtual std::string name() const = 0;
  };

  KtRecom* getRecomScheme(int recom);


  class KtRecomE : public KtRecom {    
  public:
    KtRecomE();
    virtual ~KtRecomE(){};
    /** Return merged 4-momentum */
    void combine( KtJetInfo * jet,  KtJetInfo * jet_toadd) ;
    /** Name of scheme */
    std::string name() const;
  private:
    std::string m_name;
  };

  class KtRecomPt : public KtRecom {    
  public:
    KtRecomPt();
    virtual ~KtRecomPt(){};
    /** Return merged 4-momentum */
    void combine( KtJetInfo * jet,  KtJetInfo * jet_toadd) ;
    /** Name of scheme */
    std::string name() const;
  private:
    std::string m_name;
  };


  // *****************************************************************  
  // Kt Alg
  // *****************************************************************
  class KtAlgo {
  public :
    enum KtAlgoType {
      Standard = 0,
      Aachen,
      Reversed,
      StandardExcluN,
      StandardExcluD,
      Hull
    };
    
    virtual ~KtAlgo(){};
    
    virtual void init(jetcollection_t *constituents) = 0;
    virtual bool continueClustering() = 0;    
    virtual void endClustering() = 0;    

    virtual std::string name() = 0;
    virtual void print_info() = 0;
    
    virtual void setDistanceType(KtDistance::KtDistanceType distType) = 0;
    virtual void setRecomType(KtRecom::KtRecomType recomType) = 0;    
    virtual void setProperty(std::string name, float val) {}

    // Retrieve information at a given step of clustering : 
    bool doExtractJet() {return m_doExtractJet;} ;
    KtJetInfo * getExtractJet() {return m_extractedJet;};
    float getLastDPair(){return m_lastDPair;};
    float getLastDJet(){return m_lastDJet;};
    int getLastJetIndex() { return m_lastIndex1;}
    int getLastJetIndex2() { return m_lastIndex2;}
    void setJetCollection(KtJetInfo_list *final, KtJetInfo_list *reject){m_finalJets=final;m_rejectedJets=reject;}
    
  protected:
    double m_rParameterSq;

    // at each step the following information can be calculated:
    bool m_doExtractJet;
    KtJetInfo * m_extractedJet;
    int m_lastIndex1,m_lastIndex2;
    float m_lastDPair;
    float m_lastDJet;
    
    KtJetInfo_list * m_finalJets;
    KtJetInfo_list * m_rejectedJets;

  };

  // ------------------------------------
  KtAlgo * getKtAlgo(int type, double rparam);
  // ------------------------------------
  class KtAlgoStandard : public KtAlgo {
  public:
    KtAlgoStandard() ;
    KtAlgoStandard(double rParameter) ;
    ~KtAlgoStandard();

    virtual void init(jetcollection_t *constituents) ;
    virtual bool continueClustering();
    virtual void endClustering();    

    virtual void setDistanceType(KtDistance::KtDistanceType distType) ;
    virtual void setRecomType(KtRecom::KtRecomType recomType) ;

    virtual std::string name() {return "KtAlgoStandard";};
    void print_info() {};


  protected :

    void processStep();

    KtDistance * m_ktDist;
    KtRecom * m_ktRecom;
    KtLists * m_ktList;
  };
  // ------------------------------------
  class KtAlgoReversed : public KtAlgoStandard {
  public:
    KtAlgoReversed(double rParameter)  : KtAlgoStandard() {m_rParameterSq=( 1/(rParameter*rParameter) );};

    virtual bool continueClustering();
    virtual void endClustering(); 
    virtual std::string name() {return "KtAlgoReversed";};
    virtual void setDistanceType(KtDistance::KtDistanceType /*distType*/) ;
    
  protected :
    void processStep();
  };
  // ------------------------------------
  class KtAlgoStandardExclN : public KtAlgoStandard {
  public:
    KtAlgoStandardExclN(double rparam, int njet) : KtAlgoStandard(rparam) {m_Njet=njet;}
    virtual std::string name() {return "KtAlgoStandardExclN";};
    virtual bool continueClustering();
    virtual void endClustering(); 
    void setNJet(int n){m_Njet = n;}
  protected :
    int m_Njet;
  };
  // ------------------------------------
  class KtAlgoStandardExclD : public KtAlgoStandard {
  public:
    KtAlgoStandardExclD(double rparam, double ycut) : KtAlgoStandard(rparam) {m_Ycut=ycut;}
    virtual std::string name() {return "KtAlgoStandardExclD";};
    virtual bool continueClustering();
    virtual void endClustering(); 
    void setYCut(double y){m_Ycut = y;}
  protected :
    double m_Ycut;
  };

}

}  // namespace SpartyJet
#endif
