
#ifndef FASTKTJET_KTUTILS_H
#define FASTKTJET_KTUTILS_H

#ifndef STD_STRING_H
#include <string>
#define STD_STRING_H
#endif

#include "compatibility.h"
//#include "CLHEP/Vector/LorentzVector.h"
#include "JetCore/LorentzVector.hh"
#include "JetCore/Jet.hh"
//#include "JetUtils/JetCollectionHelper.h"
#include <map>
#include <list>

#include "FastKtParam.h"

/**
 * Collection of utilities classes
 * for a fast Kt algorithm
 *
 * strongly inspired from the KtJet package
 
 @author Pierre-Antoine Delsart
*/


namespace SpartyJet { 
//class HepLorentzVectror;
class Jet;
namespace FastKtJet {
  //typedef JetCollectionHelper::jetcollection_t jetcollection_t;
  typedef Jet::jet_list_t jetcollection_t;


  // Extended Info for jets : -----------------------------------------------------
  class KtJetInfo;
  
  typedef std::list<KtJetInfo*>                     KtJetInfo_list;
  typedef KtJetInfo_list::iterator                  KtJetInfo_iterator;
  typedef std::multimap<float,KtJetInfo_iterator>   KtDist_map;
  
  class KtJetInfo{
    /// Holds all information related to a proto-jet
    /// - 4vector, pre-calculated eta,phi and pt2 for efficiency
    /// - list of constituents
    /// - pointers to position in Di and Dij lists and to NN

  public: 
    KtJetInfo(Jet * vec) ;
    KtJetInfo(float etai, float phii, float pt2i):eta(etai), phi(phii), pt2(pt2i){ };
    virtual ~KtJetInfo(){};
    int index;
    HepLorentzVector hlv;
    float eta;
    float phi;
    float pt2;

    KtJetInfo_iterator closest_neighb;      
    KtDist_map::iterator pos_in_diList;
    KtDist_map::iterator pos_in_dijList;    
    std::list<Jet *> constit_list;  // using std::slist here should be faster...
  };

  inline KtJetInfo::KtJetInfo(Jet * vec) {

    if(vec->e() <= 0.0) {
      // build a massless, ~0 energy vector in the same direction as input. 
      // this way it is correctly placed and the input Jet is still kept in constit_list
      hlv.SetVectMag(vec->Vect().Unit() *10e-10, 0); 
    }
    else {
      hlv.SetPx(vec->px());hlv.SetPy(vec->py());hlv.SetPz(vec->pz());hlv.setE(vec->e());
    }
    eta = hlv.rapidity(); phi= hlv.phi(); pt2 = hlv.Perp2(); constit_list.push_back(vec); 
  };
  // Neighborhood -------------------------------------------------
  // 
  class KtNNOperation {
    /// This class is designed to compute all Neighboring operation
    /// 
  public:
    KtNNOperation(){m_dist = 0; m_allpoints = 0;};

    /// 'list' is the eta-sorted list od proto-jets on which NN computation, insertion and removal of jets will be made
    void init(KtJetInfo_list * list, KtDistance* dist) {m_dist = dist ; m_allpoints = list;}

    /// insert/removePoint function are used to compute
    /// the list of points needing update after insertion/removal of points
    /// this list is stored in m_points_toupdate
    /// points in this list may have new di and dij value and new NN : they must be recomputed and lists reordered
    /// insertPoint actually updates NN information so no  NN recomputation is needed
    void insertPoint(KtJetInfo_iterator  jet);
    void removePoint(KtJetInfo_iterator  jet);
    void removePoint(KtJetInfo_iterator  jet1,KtJetInfo_iterator  jet2);

    KtJetInfo_iterator computeNN(KtJetInfo_iterator  jet);

    // this should not be public. Rather provide method to begin() and end()
    std::list<KtJetInfo_iterator> m_points_toupdate;

    // debugging 
    void dumpUpdatelist();
    float getlastminD(){return m_minD;}

  protected:
    struct lineq_t { float cx; float cy; float cc;} ;
    lineq_t m_d[2];
    bool m_needline;
    float m_phiRef,m_etaRef; // holding the position of the point being inserted/removed
    
    float m_minD; // used during NN computation to hold the min distance found so far

    KtJetInfo_list *m_allpoints;
    KtDistance *m_dist;
    KtJetInfo_iterator m_NN;

    /// set-up the system for removal/insertion at 'jet'
    void initPoint(KtJetInfo_iterator  jet, bool doclear = true);
    /// Internal function that fills m_points_toupdate for insertion/removal at 'jet'
    void findNNrelation(KtJetInfo_iterator jet,  bool removingMode );

    /// phi coordinate in a frame centered on m_phiRef, taking into account 
    /// clyindrical effect
    inline float translatePhi(float phi);

    inline bool lineExcluded(bool i,float eta, float phi);
    inline bool isClosest(float eta, float phi);
    void update_lines(bool i ,float eta, float phi, bool dosym );
    
    // debugging 
    inline float lineExcludedV(bool i,float eta, float phi);

  };
  

  // KtJetLists --------------------------------------------------------------------  
  //
  class KtLists   {
    /// Holds and performs merging or removing operations on proto-jets list 
    /// It also holds pointer lists and uses a KtNNoperation operator
    /// to be as efficient as possible

  public:
    KtLists(){};
    KtLists(jetcollection_t *, KtDistance *, KtRecom *recom, bool reversed_mode =false);
    virtual ~KtLists();
    /* Number of in list jets */
    inline int getNJets() const;                        

    /** get jet with min  KtDistance with respect to beam */
    inline KtJetInfo * getMinJet() const;    
    inline KtJetInfo * getMaxJet() const;    
    /** get jet with min  KtDistance to its neighbour. One access the
	nn from this jet */
    inline KtJetInfo * getMinPairJet() const;    
    inline KtJetInfo * getMaxPairJet() const;    

    /** get min  KtDistance with respect to beam */
    inline float getMinDJet() const;                       
    /** get min KtDistance of pair of jets  */
    inline float getMinDPair() const;                       
    /** get index of of jet with min KtDistance with respect to beam */
    inline int getMinJetIndex() const;
    /** get indices of pair with min KtDistance */
    inline std::pair<int , int > getMinPairIndex() const;


    // Needed for reversed Kt :
    /** get max  KtDistance with respect to beam */
    inline float getMaxDJet() const;                       
    /** get max KtDistance of pair of jets  */
    inline float getMaxDPair() const;                       
    /** get index of of jet with max KtDistance with respect to beam */
    inline int getMaxJetIndex() const;
    /** get indices of pair with max KtDistance */
    inline std::pair<int , int > getMaxPairIndex() const;

    // Generic getter at each steps of the KtLoop
    /** get step  KtDistance with respect to beam */
    inline KtJetInfo * getStepJet() const {if (m_reversedMode) return getMaxJet(); else return getMinJet();}    
    inline float getStepDJet() const {if (m_reversedMode) return getMaxDJet(); else return getMinDJet();}; 
    /** get step KtDistance of pair of jets  */
    inline float getStepDPair() const {if (m_reversedMode) return getMaxDPair(); else return getMinDPair();}; 
    /** get index of of jet with step KtDistance with respect to beam */
    inline int getStepJetIndex() const {if (m_reversedMode) return getMaxJetIndex(); else return getMinJetIndex();}; 
    /** get indices of pair with step KtDistance */
    inline std::pair<int , int > getStepPairIndex() const {if (m_reversedMode) return getMaxPairIndex(); else return getMinPairIndex();}; 


    /** Combine jets  (E-scheme only so far) */ 
    inline void mergeMinJets();                  
    inline void mergeMaxJets();     // Needed for reversed Kt                   
    void mergeJets(KtJetInfo_iterator jeti, KtJetInfo_iterator jetj);           
    /** Delete jet  from table */
    inline void killMinJet();
    inline void killMaxJet();     // Needed for reversed Kt 
    inline void killMinPairFirst();

    void killJet(KtJetInfo_iterator jeti);

    inline void killStepJet() {if (m_reversedMode) killMaxJet(); else killMinJet(); };     
    inline void mergeStepJets() {if (m_reversedMode)  mergeMaxJets() ; else mergeMinJets();};  
    float getERemaining();

    static bool  KtJetInfoEtaPhiComp(KtJetInfo* k1, KtJetInfo* k2) {    return  ( (k1->eta - k2->eta ) < 0) ;};
  protected:
    bool m_reversedMode;
    
    /** Initial number of jets/particles */
    int m_nRemaining;

    /** Function object to define Kt distance scheme */
    KtDistance *m_fKtDist;
    /** Recombination scheme */
    KtRecom *m_ktRecom;

    /// Neigboorhood : 
    KtNNOperation* m_NNoperator;
    

    /** list of Kt distances respect to beam */
    KtDist_map m_diList;

    /** list of Kt distances  with respect to nearest neighboor */
    KtDist_map m_dijList;
  
    /** list of (jets+info) */
    KtJetInfo_list m_ktlist;





    virtual void buildKtJetInfo(jetcollection_t *jetColl);
    inline KtDist_map::iterator insertValueAndIter(KtDist_map &themap, float thekey, KtJetInfo_iterator theit);
    inline KtDist_map::iterator insertValueAndIter_withHint(KtDist_map &themap, float thekey, KtJetInfo_iterator theit, KtDist_map::iterator hint);
    inline void removefromDLists(KtJetInfo*);
    
    // debugging
    bool checkddi(std::string s , KtDist_map::iterator  ddit);
    void checkddi();
    void dumpddi();
    void checkdGi();
    void checkNN() ;
    void dumpktlist();
  };
 




  // ***************************** inlined functions *************************************
  inline int KtLists::getNJets() const {
    return m_nRemaining;
  }

  inline void KtLists::removefromDLists(KtJetInfo* jet){
    m_diList.erase(jet->pos_in_diList); 
    m_dijList.erase(jet->pos_in_dijList);
  }
 
 
  inline float KtLists::getMinDJet() const {
    return (*(m_diList.begin())).first;
  }
  inline int KtLists::getMinJetIndex() const {
    return (*(*(m_diList.begin())).second)->index;
  }
  inline KtJetInfo  * KtLists::getMinJet() const {
    KtJetInfo_iterator jeti = (*(m_diList.begin())).second;
    return (*jeti);
  }
  inline KtJetInfo  * KtLists::getMinPairJet() const {
    KtJetInfo_iterator jeti = (*(m_dijList.begin())).second;
    return (*jeti);
  }


  inline float KtLists::getMinDPair() const {
    return (*(m_dijList.begin())).first;
  }
  inline std::pair<int, int> KtLists::getMinPairIndex() const {
    KtJetInfo_iterator jeti = (*(m_dijList.begin())).second;
    return std::pair<int,int>((*jeti)->index, (*(*jeti)->closest_neighb)->index);
  }

  inline float KtLists::getMaxDJet() const {
    return (*(--m_diList.end())).first;
  }
  inline int KtLists::getMaxJetIndex() const {
    return (*(*(--m_diList.end())).second)->index;
  }
  inline KtJetInfo  * KtLists::getMaxJet() const {
    KtJetInfo_iterator jeti = (*(--m_diList.end())).second;
    return (*jeti);
  }
  inline KtJetInfo  * KtLists::getMaxPairJet() const {
    KtJetInfo_iterator jeti = (*(--m_dijList.end())).second;
    return (*jeti);
  }

  inline float KtLists::getMaxDPair() const {
    return (*(--m_dijList.end())).first;
  }
  inline std::pair<int, int> KtLists::getMaxPairIndex() const {
    KtJetInfo_iterator jeti = (*(--m_dijList.end())).second;
    return std::pair<int,int>((*jeti)->index, (*(*jeti)->closest_neighb)->index);
  }


  inline void KtLists::mergeMinJets() {  
    // get Jets : 
    KtDist_map::iterator mindij = m_dijList.begin();
    KtJetInfo_iterator jeti = (*mindij).second;
    KtJetInfo_iterator jetj = (*jeti)->closest_neighb;
    mergeJets(jeti, jetj);
  }
  inline void KtLists::mergeMaxJets() {  
    // get Jets : 
    KtDist_map::iterator mindij = m_dijList.end(); mindij--;
    KtJetInfo_iterator jeti = (*mindij).second;
    KtJetInfo_iterator jetj = (*jeti)->closest_neighb;
    mergeJets(jeti, jetj);
  }

  inline void KtLists::killMinJet() {
    // m_diList is sorted so the jet we want is the first of it :
    KtJetInfo_iterator jeti = (*(m_diList.begin())).second;
    killJet(jeti);
  }
  inline void KtLists::killMaxJet() {
    // m_diList is sorted so the jet we want is the last of it :  
    KtDist_map::iterator it=m_diList.end(); it--;
    KtJetInfo_iterator jeti = (*(it)).second; 
    killJet(jeti);
  }

  inline void KtLists::killMinPairFirst() {
    // m_diList is sorted so the jet we want is the first of it :
    KtJetInfo_iterator jeti = (*(m_dijList.begin())).second;
    killJet(jeti);
  }



  class KtNNOperationPhiBin : public KtNNOperation {
    // not implemented. 
    // Using a binned (eta,phi) plan may improve speed...

  };


  
  inline KtDist_map::iterator KtLists::insertValueAndIter_withHint(KtDist_map &themap, float thekey, KtJetInfo_iterator theit,KtDist_map::iterator hint){
    std::pair<float,KtJetInfo_iterator> p(thekey , theit);
    return themap.insert(hint,p);
  }
  inline KtDist_map::iterator KtLists::insertValueAndIter(KtDist_map &themap, float thekey, KtJetInfo_iterator theit){
    std::pair<float,KtJetInfo_iterator> p(thekey , theit);
    return themap.insert(p);    
  } 


}// end of namespace

}  // namespace SpartyJet
#endif
