#ifndef CELLJET_HH
#define CELLJET_HH

#include "CellJetEvent.hh"
namespace SpartyJet { 

namespace Pythia8 {

//**************************************************************************

// SingleCell class.
// Simple helper class to CellJet for a cell and its contents. 

class SingleCell {

public:

  // Constructor.
  SingleCell(int iCellIn = 0, double etaCellIn = 0., double phiCellIn = 0., 
    double eTcellIn = 0., int multiplicityIn = 0) : iCell(iCellIn), 
    etaCell(etaCellIn), phiCell(phiCellIn), eTcell(eTcellIn), 
    multiplicity(multiplicityIn), canBeSeed(true), isUsed(false),
    isAssigned(false) {}

  // Properties of cell.
  int iCell;
  double etaCell, phiCell, eTcell;
  int multiplicity;
  bool canBeSeed, isUsed, isAssigned;

} ;

//**************************************************************************


// SingleCellJet class.
// Simple helper class to CellJet for a jet and its contents. 

class SingleCellJet {

public:

  // Constructor.
  SingleCellJet(double eTjetIn = 0., double etaCenterIn = 0., 
    double phiCenterIn = 0., double etaWeightedIn = 0.,
    double phiWeightedIn = 0., int multiplicityIn = 0,
    Vec4 pMassiveIn = 0.) : eTjet(eTjetIn), etaCenter(etaCenterIn), 
    phiCenter(phiCenterIn), etaWeighted(etaWeightedIn), 
    phiWeighted(phiWeightedIn), multiplicity(multiplicityIn),
    pMassive(pMassiveIn) {}

  // Properties of jet.
  double eTjet, etaCenter, phiCenter, etaWeighted, phiWeighted;
  int multiplicity;
  Vec4 pMassive;

} ;

//**************************************************************************

// CellJet class.
// This class performs a cone jet search in (eta, phi, E_T) space.

class CellJet {

public: 

  // Constructor.
  CellJet(double etaMaxIn = 5., int nEtaIn = 50, int nPhiIn = 32, 
    int selectIn = 2, int smearIn = 0, double resolutionIn = 0.5, 
    double upperCutIn = 2., double thresholdIn = 0.) : etaMax(etaMaxIn), 
    nEta(nEtaIn), nPhi(nPhiIn), select(selectIn), smear(smearIn),
    resolution(resolutionIn), upperCut(upperCutIn), 
    threshold(thresholdIn) { }
  
  // Analyze event.
  bool analyze(CellJetEvent& event, double eTjetMinIn = 20., 
    double coneRadiusIn = 0.7, double eTseedIn = 1.5);

  // Return info on results of analysis.
  int size() const {return jets.size();}
  double eT(int i) const {return jets[i].eTjet;}
  double etaCenter(int i) const {return jets[i].etaCenter;}
  double phiCenter(int i) const {return jets[i].phiCenter;}
  double etaWeighted(int i) const {return jets[i].etaWeighted;}
  double phiWeighted(int i) const {return jets[i].phiWeighted;}
  double multiplicity(int i) const {return jets[i].multiplicity;}
  Vec4 pMassless(int i) const {return jets[i].eTjet * Vec4(
    cos(jets[i].phiWeighted), sin(jets[i].phiWeighted),
    sinh(jets[i].etaWeighted), cosh(jets[i].etaWeighted) );}
  Vec4 pMassive(int i) const {return jets[i].pMassive;}
  double m(int i) const {return jets[i].pMassive.mCalc();}
  vector<SingleCellJet> get_jets() {return jets;}

  // Provide a listing of the info.
  void list(ostream& os = cout);

private: 

  // Properties of analysis.
  double etaMax; 
  int nEta, nPhi, select, smear;
  double resolution, upperCut, threshold;
  double eTjetMin, coneRadius, eTseed; 

  // Outcome of analysis: ET-ordered list of jets. 
  vector<SingleCellJet> jets;

};

} // end namespace Pythia8

}  // namespace SpartyJet
#endif
