import os
import ROOT
_v = ROOT.TLorentzVector()

_sparty_lib = os.environ.setdefault('SPARTYJETLIBS','../libs')

ROOT.gSystem.Load(_sparty_lib+'/libJetTools.so')

from ROOT import SpartyJet as SJ
import JetHistoryGui

_initialCutTool = SJ.JetPtSelectorTool(0.0)
_finalCutTool = SJ.JetPtSelectorTool(1.0)

class AlgoConfig(object):
    """ Configuration base class for live algorithm
    These are functor objects : they emulate a function by defining the
    __call__ method (see below).
    This function call returns a configured JetTool
    """
    name=""
    @staticmethod
    def rtostr(r):
        r_str = str(round(r*10))
        r_str = r_str.replace('.','')
        if r_str[1] == '0': r_str = r_str[:-1]
        return r_str

    def __call__(self, r , *args):
        """ returns a configured instance of a JetTool implementing a jet 
        finding algorithm.
        r is the 'main parameter' (typically radius)
        *args contains 2 other optionnal parameter.
        Details of configuration depend on the inherited class
        """
        pass


class AntiKtConf(AlgoConfig):
    name="AntiKt"
    def __call__(self, r , *args):
        ROOT.gSystem.Load(_sparty_lib+"/libFastJet.so") 
        r_str = AlgoConfig.rtostr(r)
        fj = SJ.FastJet.FastJetFinder('AntiKt'+r_str)
        fj.set_algorithm(2)
        fj.set_Rparam(r)
        fj.set_area(False)
        fj.set_history_building(True)
        return fj

class KtConf(AlgoConfig):
    name="Kt"
    def __call__(self, r , *args):
        ROOT.gSystem.Load(_sparty_lib+"/libFastJet.so") 
        r_str = AlgoConfig.rtostr(r)
        fj = SJ.FastJet.FastJetFinder('Kt'+r_str)
        fj.set_Rparam(r)
        fj.set_history_building(True)
        return fj

class CambridgeConf(AlgoConfig):
    name="Cambridge"
    def __call__(self, r , *args):
        ROOT.gSystem.Load(_sparty_lib+"/libFastJet.so") 
        r_str = AlgoConfig.rtostr(r)
        fj = SJ.FastJet.FastJetFinder('Cam'+r_str)
        fj.set_Rparam(r)
        fj.set_algorithm(1)
        return fj

class SISConeConf(AlgoConfig):
    name="SISCone"
    def __call__(self, r , *args):
        ROOT.gSystem.Load(_sparty_lib+"/libFastJet.so") 
        r_str = AlgoConfig.rtostr(r)
        sc = SJ.FastJet.SISConeFinder('SISCone'+r_str)
        sc.set_coneRadius(r)
        return sc

# All known algorithm :
defined_algs = [
    AntiKtConf(),
    KtConf(),
    CambridgeConf(),
    SISConeConf(),
]



class PyJetRetrieverLive(object):    
    """Holds a JetAlgorithm and uses it to rebuild a jet collection"""

    @staticmethod
    def setFinalCut(cut):
        _finalCutTool.setCut(cut)

    def __init__(self, jettool):
        """ jettool  must be a jet finding JetTool"""
        jetalg = SJ.JetAlgorithm(jettool.name())
        jetalg.addTool(_initialCutTool)
        jetalg.addTool(jettool)
        jetalg.addTool(_finalCutTool)
        self.jetalg = jetalg
        self.jettool = jettool
        jetalg.init()
        self.currentEvent = -1
        pass


    def fill_collection(self, jc_item, event):
        """ jc_item a JetCollItems. 
        The input jet collection and jet collection associated (jetcoll) to this jc_item are filled
        jc_item.jetcoll is the result of the jet algorithm holded by self
        event (int) : the event number
        """
        # do not re-run if this event has already been found.
        if self.currentEvent == event:
            return
        self.currentEvent = event
        jc_item.jetcoll.clear_and_delete()
        print ' Filling collection ',jc_item.inputcoll , jc_item.jetcoll
        jc_item.input_retriever.fill_inputColl(jc_item.inputcoll,event)
        print '  ... input filled ',jc_item.inputcoll.size()
        self.jetalg.execute(jc_item.inputcoll, jc_item.jetcoll)
        print '  JetRetrieverLive alg executed.'
        self.jetalg.finalize()
        ## inph = set( el.jet for el in JetHistoryGui.vector_iterate(jc_item.jetcoll.history()) if not el.parent1  )
        ## for  i in inph:
        ##     print i, i.e()
        #print jc_item.jetcoll,'  after fill_collection inph=',inph

        jc_item.jetcoll.set_algo_indices(jc_item.inputcoll.size())

    def get_current_event(self):
        return self.currentEvent
    

