#include "JetCore/JetBuilder.hh"
#include "JetCore/HepMCInput.hh"
#include "JetCore/JetAlgorithm.hh"
#include "fastjet/FastJetFinder.hh"
#include "fastjet/SISConeFinder.hh"
#include "fastjet/YSplitterTool.hh"
#include "JetTools/EConversionTool.hh"
#include "JetTools/EtaPhiMomentTool.hh"
#include "JetTools/HullMomentTool.hh"
#include "JetTools/JetSelectorTool.hh"
#include "eventshape/ThrustTool.hh"

//fastjet includes
#include "fastjet/AreaDefinition.hh"

using namespace SpartyJet;
using namespace SpartyJet::FastJet;

int main(){
  
  // Create a builder object
  JetBuilder builder;
	builder.set_message_level(INFO);

	// Create an input object and open the input file
	HepMCInput input("../data/HepMC_sample.dat");
  builder.configure_input((InputMaker*)&input);
  
  // Configure algorithms
  FastJetFinder *antikt4 = new FastJetFinder("AntiKt4",fastjet::antikt_algorithm,0.4,true);
	antikt4->set_areaChoice(fastjet::active_area);
  builder.add_default_alg(antikt4);
  FastJetFinder *kt4 = new FastJetFinder("Kt4",fastjet::kt_algorithm,0.4,true);
	kt4->set_areaChoice(fastjet::active_area);
  builder.add_default_alg(kt4);
  SISConeFinder *siscone4 = new SISConeFinder("SISCone4",0.4,true);
	siscone4->set_areaChoice(fastjet::passive_area);
  builder.add_default_alg(siscone4);
  
	// Add Jet Tools ---------------------------------
	std::vector<int> pdgIdsToIgnore;
	pdgIdsToIgnore.push_back(12); 
	pdgIdsToIgnore.push_back(-12);
	builder.add_jetTool_front(new JetInputPdgIdSelectorTool(pdgIdsToIgnore));
	builder.add_jetTool_front(new JetPtSelectorTool(10),"SISCone4");
	builder.add_jetTool_front(new JetPtSelectorTool(1));
	builder.add_jetTool_front(new JetEtaCentralSelectorTool(-1.5,1.5));

	//======Here is where the algorithms defined above will run ======

	builder.add_jetTool(new JetPtSelectorTool(7));
	builder.add_jetTool(new YSplitterTool(0.1,fastjet::kt_algorithm,2));
	builder.add_jetTool(new EtaPhiMomentTool("EtaPhiMomTool"));
	builder.add_jetTool(new HullMomentTool("HullMomTool"));
	builder.add_jetTool(new EConversionTool("EConvTool", false)); // Convert to MeV

	// Add Event Shape -------------------------------
	SpartyJet::JetAlgorithm *thrustAlg = new SpartyJet::JetAlgorithm("thrustAlg");
	thrustAlg->addTool(new eventshape::ThrustTool("ThrustTool"));
	builder.add_eventshape_alg(thrustAlg);
  
	// Configure the output (name of tree, root file)
  builder.configure_output("SpartyJet_Tree","../data/output/simple.root");
  
  // Run the builder on the first 10 events
  builder.process_events(10);

  return 0;
}
