#include "fastjet/PseudoJet.hh"
#include "fastjet/ClusterSequenceActiveArea.hh"
#include "fastjet/AreaDefinition.hh"
#include "fastjet/ClusterSequenceArea.hh"
#include "fastjet/ClusterSequencePassiveArea.hh"
#include "fastjet/ClusterSequence.hh"
#include "fastjet/ActiveAreaSpec.hh"

#include "JetCore/CommonUtils.hh"
#include "FastJetUtils.hh"
#include <vector>
#include "FastJetFinder.hh"

using namespace std;

namespace SpartyJet { 

namespace FastJet {

FastJetFinder::FastJetFinder(string name,fastjet::JetAlgorithm alg,double R,bool area)
                            : JetTool(name) {
	set_defaults(); // sets all other parameters to sensible values
  set_algorithm(alg, R);
  m_area = area;
  init();
}

FastJetFinder::FastJetFinder(fastjet::JetDefinition *jet_def, string name, bool area)
                            : JetTool(name) {
  set_defaults();
  m_use_ext_jet_def = true;
  m_jet_def = jet_def;
  m_area = area;
  init();
}

FastJetFinder::~FastJetFinder()
{
  if (! m_use_ext_jet_def) // only delete jet_def if we made it
	delete m_jet_def; m_jet_def = NULL;
}

void FastJetFinder::set_defaults() {
  m_jet_def = NULL;
  m_use_ext_jet_def = false;

  m_areaChoice    = fastjet::active_area;
  m_inclusive     = true;
  m_exclusive     = false;

  m_ptmin         = 5.0;
  m_dcut          = 25.0;

  m_ghost_etamax  = 6.0;
  m_repeat        = 5;
  m_ghost_area    = 0.01;
  m_grid_scatter  = 1E-4;
  m_kt_scatter    = 0.01;
  m_mean_ghost_kt = 1E-100;
}

void FastJetFinder::configure(fastjet::JetAlgorithm a,bool ar,bool in,bool ex,double r,double p,double d) {
  set_algorithm(a,r);
  m_area      = ar;
  m_inclusive = in;
  m_exclusive = ex;
  m_ptmin     = p;
  m_dcut      = d;

  m_areaChoice= fastjet::active_area;
  init();
}

// Should we bother safety check here?  Let FastJet figure it out...
void FastJetFinder::set_algorithm(fastjet::JetAlgorithm x, double R) {
  if(x != fastjet::kt_algorithm && x != fastjet::cambridge_algorithm && x != fastjet::antikt_algorithm 
		&& x != fastjet::genkt_algorithm && x != fastjet::cambridge_for_passive_algorithm 
		&& x != fastjet::genkt_for_passive_algorithm) {
    cout << "FastJet algorithm \"" << x << "\" is not defined." << endl;
    cout << "acceptable algorithms are: " << endl;
    cout << " kt  : kt_algorithm " << endl;
    cout << " CambAach  : cambridge_algorithm " << endl;
    cout << " antikt  : antikt_algorithm " << endl;
    cout << " genkt  : genkt_algorithm " << endl;
    cout << " CambAachPassive : cambridge_for_passive_algorithm " << endl;
    cout << " genktPassive : genkt_for_passive_algorithm " << endl;
  }
  else if (m_use_ext_jet_def) {
    cout << "Using external jet definition, don't call set_algorithm -- mess with it yourself!" << endl;
  }
  else {
    // Prepare JetDefinition
    if (m_jet_def != NULL) delete m_jet_def;
    m_jet_def = new fastjet::JetDefinition((fastjet::JetAlgorithm)x, R);
    m_log << INFO << "FastJet configured to run: " << m_jet_def->description() << endl;
  }
}

void FastJetFinder::configure_area(double e,int r,double a,
                                   double s,double ks,double kt) {
  m_ghost_etamax  = e;
  m_repeat        = r;
  m_ghost_area    = a;
  m_grid_scatter  = s;
  m_kt_scatter    = ks;
  m_mean_ghost_kt = kt;
}



void FastJetFinder::init(JetMomentMap *mmap) {
  if((m_area) && (mmap != NULL)) {
    mmap->schedule_jet_moment("area");
    mmap->schedule_jet_moment("area_error");
  }
}


void FastJetFinder::execute(JetCollection &inputJets) {

  JetCollection tmp_list;
  
	// retrieve the map of the collection :
  JetMomentMap * themap = tmp_list.get_JetMomentMap();
	// Copy Jet moment layout from input
	themap->copy_structure(*(inputJets.get_JetMomentMap()));
  
  // Prevent problems with empty events :
  if (inputJets.size() == 0) return;

  // convert inputJets to psuedojets
  JetCollection::iterator iter = inputJets.begin();
  vector<fastjet::PseudoJet> pjets;
  pjets.reserve(inputJets.size()); // we know what will be the size
  Jet *tjet = NULL;
	int ind=0;
  while(iter != inputJets.end()){
    tjet = *iter;
      pjets.push_back(fastjet::PseudoJet(*(*iter)));
			// we must reset the index according to inputJets (the original in *iter is potentially not in sinc with the
			// inputJets if the original colection has been filtered).
    pjets.back().set_user_index(ind);
		++iter;++ind;
  }

  fastjet::ClusterSequence * clust_seq = 0;
  fastjet::ClusterSequenceArea * clust_seq_area = 0; // used only if area requested

  int areapos = 0;
  int area_errorpos = 0;

  // Find jets with or without area ----------------------------------------------
  if(m_area) {          // area calculation turned on
		fastjet::AreaDefinition area_def;
    if(m_areaChoice == fastjet::voronoi_area) {
      fastjet::VoronoiAreaSpec Varea_spec;
      area_def = fastjet::AreaDefinition(Varea_spec);
    } else {
      fastjet::GhostedAreaSpec Garea_spec = fastjet::GhostedAreaSpec(m_ghost_etamax,m_repeat,m_ghost_area,m_grid_scatter,m_kt_scatter,m_mean_ghost_kt);
      area_def = fastjet::AreaDefinition(Garea_spec,(fastjet::AreaType)m_areaChoice);
    }
    clust_seq_area = new fastjet::ClusterSequenceArea(pjets, *m_jet_def,area_def);
    
    clust_seq     = clust_seq_area;
    areapos       = themap->get_jet_momentPos("area");
    area_errorpos = themap->get_jet_momentPos("area_error"); 
    
  } else {
    clust_seq = new fastjet::ClusterSequence(pjets, *m_jet_def); 
  }


  vector<fastjet::PseudoJet> output_jets;
  // Retrieve jets ----------------------------------------------
  if(m_inclusive && !m_exclusive) {                                         // only inclusive
      output_jets = sorted_by_pt(clust_seq->inclusive_jets(m_ptmin));
  }
  if(m_exclusive && !m_inclusive){                                          // only exclusive
    output_jets = sorted_by_pt(clust_seq->exclusive_jets(m_ptmin));
  } else if(m_exclusive && m_inclusive){                                    // both inclusive and exclusive
    vector<fastjet::PseudoJet>  temp_jets = sorted_by_pt(clust_seq->inclusive_jets(m_ptmin));
    for(unsigned i = 0; i < temp_jets.size(); i++)
      output_jets.push_back(temp_jets[i]);
    
    temp_jets.clear();
    temp_jets = sorted_by_pt(clust_seq->exclusive_jets(m_ptmin));
    for(unsigned i = 0; i < temp_jets.size(); i++)
      output_jets.push_back(temp_jets[i]);
  }

  // Convert jets to SpartyJet format ----------------------------------------------
  for(unsigned i = 0; i < output_jets.size(); i++) {
    
    fastjet::PseudoJet &jet = output_jets[i];
    
    // convert back to jet class
    Jet* t2jet = new Jet(jet.px(),jet.py(),jet.pz(),jet.E());
    
    if(m_area){
			themap->set_jet_moment(areapos,t2jet,(float)clust_seq_area->area(jet));  // setting the area for the momentmap
      themap->set_jet_moment(area_errorpos,t2jet,(float)clust_seq_area->area_error(jet));
    }
    
    // Retrieve jets' constituents
    vector<fastjet::PseudoJet> constituents = clust_seq->constituents(jet);
    for(unsigned c=0;c<constituents.size();c++){
      int index = constituents[c].user_index();
      // get original constituents of this jet (!!! there maybe more
      // than 1, following is not correct !!!)
      Jet* constit = *(inputJets[index]->firstConstituent());
      t2jet->addConstituent_notMoment(constit);
    }
    tmp_list.push_back(t2jet);
  }

  if(m_updateHistory) {
    m_log << DEBUG <<  "FastJet buildHistory"  << endl;
    const vector<fastjet::ClusterSequence::history_element> & fastjetHist = clust_seq->history();
    const vector<fastjet::PseudoJet>  alljets = clust_seq->jets();
    
    size_t Nh = fastjetHist.size();
    
    tmp_list.prepare_history(Nh);
    vector<HistoryElement> & sjHist = *tmp_list.history();
    
    for(size_t i=0;i<Nh;i++){
      HistoryElement &el = sjHist[i];
      const fastjet::ClusterSequence::history_element & fastjetel = fastjetHist[i];
      
      el.setDij( fastjetel.dij);
      el.setHindex( i );
      
      if(fastjetel.parent1 != fastjet::ClusterSequence::InexistentParent ) {
        if(fastjetel.parent1 < 0 ) cout << i <<" Wrong parent1 "<< fastjetel.parent1 << endl;
        el.setParent1( & sjHist[ fastjetel.parent1] ) ;
        //cout << i << "  ---> has parent1   "<< el.parent1() << " jet ="<<  el.jet() << endl;
      } else { // it is an input jet
        
        int sj_index_input = alljets[fastjetel.jetp_index].user_index();
        
        Jet * inputjet = inputJets[sj_index_input];
        if( inputjet->getConstituentNum() == 0 ) cout << " !!!! input with 0 constituent at  "<< i<< endl;
        el.setJet( *(inputjet->firstConstituent()) ) ; // assume the input jet is made of only 1 constituents !!
        
        //        cout << i << "  ---> constituent   "<< sj_index_input << " jet ="<<  el.jet() << endl;
      }
      if(fastjetel.parent2 >=0  ){
        el.setParent2( & sjHist[ fastjetel.parent2] ) ;
        //        cout << i << "  ---> has parent2   "<< el.parent2() << " jet ="<<  el.jet() << endl;
      }
      if(fastjetel.child != fastjet::ClusterSequence::Invalid ) el.setChild( & sjHist[ fastjetel.child] );
    }
    
    // loop over all final jets to set their jet pointer :
    Nh = output_jets.size();
    //vector<int> & history_index = *tmp_list.get_history_index();
    //history_index.resize(Nh);
    for(size_t i=0;i<Nh;i++){
      //jet = output_jets[i].cluster_sequence_history_index();
      int hind = output_jets[i].cluster_sequence_history_index();
      Jet *j = tmp_list[i];
      sjHist[hind].setJet( j );
      j->setHistoryIndex(hind);
    }
  }
  
	// we can now delete the input collection :
	clear_list(inputJets);
	// and update it :
	inputJets.swap(tmp_list);

  // // tests 
  // if(m_updateHistory) {
  //   cout << "after swap dump" << endl;
  //   inputJets.history_dump();
  // }

  // clear --------------------------------
  if(m_area){
    delete clust_seq_area;
  }else{
    delete clust_seq;
  }

}

}  // namespace SpartyJet::FastJet
}  // namespace SpartyJet

