#include "PtDensityTool.hh"
#include "JetCore/JetMomentMap.hh"
#include "FastJetUtils.hh"
#include "fastjet/ClusterSequenceArea.hh"

namespace SpartyJet { 

namespace FastJet{

PtDensityTool::PtDensityTool(std::string name,fastjet::JetAlgorithm densAlg,
											float R,fastjet::AreaType area,std::vector<float> etaBins)
: JetTool(name) {
	m_densDef  = fastjet::JetDefinition(densAlg,R);
	m_areaDef = fastjet::AreaDefinition(area);
	setEtaBins(etaBins);
}

PtDensityTool::PtDensityTool(std::string name,fastjet::JetDefinition *densDef,
															fastjet::AreaType area,std::vector<float> etaBins)
: JetTool(name) {
	m_densDef  = *densDef;
	m_areaDef = fastjet::AreaDefinition(area);
	setEtaBins(etaBins);
}

PtDensityTool::PtDensityTool(std::string name,fastjet::JetDefinition *densDef,
															fastjet::AreaType area,std::vector<fastjet::RangeDefinition> *rangeDefVec)
: JetTool(name) {
	m_densDef  = *densDef;
	m_areaDef = fastjet::AreaDefinition(area);
	m_rangeDefVec = *rangeDefVec;
}

void PtDensityTool::init(JetMomentMap *mmap) 
{
  if(mmap != NULL){
		if(mmap->get_event_moment_arrayPos("ptDensity") == -1)
		{
			mmap->schedule_event_moment_array("ptDensity");
			mmap->schedule_event_moment_array("ptDensityBins");
		}
	}
}
void PtDensityTool::execute(JetCollection &inputJets) 
{
	// retrieve the map of the collection :
	JetMomentMap * themap = inputJets.get_JetMomentMap();
	if(themap->num_jet_moment() ==0) return;

	// Convert to PseudoJets
  std::vector<fastjet::PseudoJet> pjets;
  pjets.reserve(inputJets.size()); // we know what will be the size
  JetCollection::iterator jItr = inputJets.begin();
	for(;jItr != inputJets.end();++jItr){
		pjets.push_back(fastjet::PseudoJet(*(*jItr)));
  }
	
	// Find jets
	fastjet::ClusterSequenceArea clust_seq_area(pjets,m_densDef,m_areaDef);
	
	// Extract ptDensity
	std::vector<float> rhoVec;
	std::vector<float> 	rhoBinVec;
	std::vector<fastjet::RangeDefinition>::iterator rItr = m_rangeDefVec.begin();
	std::vector<fastjet::RangeDefinition>::iterator rItrE = m_rangeDefVec.end();
	double rapMin,rapMax;
	rItr->get_rap_limits(rapMin,rapMax);
	rhoBinVec.push_back((float)rapMin);
	for(;rItr!=rItrE;++rItr)
	{
		double rho = clust_seq_area.median_pt_per_unit_area(*rItr);
		rhoVec.push_back((float)rho);
		m_log << DEBUG << "rho = " << rho << " in " << (*rItr).description() << std::endl;
		rItr->get_rap_limits(rapMin,rapMax);
		rhoBinVec.push_back((float)rapMax);
	}
	
	// Save to map
	//themap->set_event_moment_array("ptDensity", rhoVec);
	//themap->set_event_moment_array("ptDensityBins", rhoBinVec);
	themap->set_event_moment_array(0, rhoVec);
	themap->set_event_moment_array(1, rhoBinVec);
	
}

void PtDensityTool::setEtaBins(std::vector<float> etaBins)
{
	float rapMin = etaBins[0];
	for(unsigned int i=1;i<etaBins.size();++i)
	{
		m_rangeDefVec.push_back(fastjet::RangeDefinition(rapMin,etaBins[i]));
		rapMin = etaBins[i];
	}
		
}
}  // namespace SpartyJet::FastJet
}  // namespace SpartyJet
					                   	                
