#include "fastjet/YSplitterTool.hh"
#include "fastjet/PseudoJet.hh"
#include "fastjet/ClusterSequence.hh"

namespace SpartyJet { 

namespace FastJet {
  
const int  YSplitterTool::MAX;

YSplitterTool::YSplitterTool(float R, fastjet::JetAlgorithm alg, int ny, int njet, std::string name)
: JetTool(name) 
{ 
	m_jet_def = new fastjet::JetDefinition(alg, R,(fastjet::Best));
	configure(ny, njet);
}

YSplitterTool::YSplitterTool(fastjet::JetDefinition *jet_def, int ny, int njet, std::string name)
: JetTool(name) 
{ 
	m_jet_def = jet_def;
	configure(ny, njet);
}

void YSplitterTool::configure(int ny, int njet){
		
    m_njets = njet;
		
		if(ny > MAX) {
      m_log << ERROR << "MAX number of values is " << MAX<< ", setting n="<< MAX << std::endl;
      m_nvalues = MAX;
    }
    else
      m_nvalues = ny;
}

void YSplitterTool::init(JetMomentMap *mmap){
  if(mmap != NULL){
    mmap->schedule_jet_moment_array("YS_vals");
  }
}

void YSplitterTool::execute(JetCollection &inputJets){
  
	// retrieve the map of the collection :
  JetMomentMap * themap = inputJets.get_JetMomentMap();
  if(themap->num_jet_moment_array() ==0) return;
  int mom_pos= themap->get_jet_moment_arrayPos("YS_vals");
  if(mom_pos <0){
    m_log << ERROR << "No scheduled moment YS_vals"<< std::endl;
    return;
  }

  // Loop over jets and run alg on constituents
  JetCollection::iterator jit = inputJets.begin(); 
  JetCollection::iterator jitE = inputJets.end(); 
  int nFilled=0;
  for(; jit!=jitE;++jit,++nFilled){
  	JetMomentMap::value_store_t arr(m_nvalues);
    
		if(nFilled >= m_njets && m_njets != -1){ 
      // do nothing. just fill the moment with blank array
      themap->set_jet_moment_array(mom_pos,(*jit),arr);
    }

    double jet_scale;;
    if(m_jet_def->jet_algorithm() == fastjet::kt_algorithm) 
			jet_scale = (*jit)->perp2();
		else if(m_jet_def->jet_algorithm() == fastjet::cambridge_algorithm) 
			jet_scale = 1.0;
		else if(m_jet_def->jet_algorithm() == fastjet::antikt_algorithm)
			jet_scale = 1/(*jit)->perp2();
		else
			jet_scale = (*jit)->perp2();
			
    // Get constituents
		std::vector<fastjet::PseudoJet> constituents;
		Jet::constit_vect_t::iterator it  = (*jit)->firstConstituent();
		Jet::constit_vect_t::iterator itE = (*jit)->lastConstituent();
		for(; it != itE;++it){
			constituents.push_back(fastjet::PseudoJet(*(*it)) );
		}
    // Recluster constituents with new jet def
		fastjet::ClusterSequence clust_seq(constituents, *m_jet_def);
    // Extract Y values
		for (int i =1; i<m_nvalues+1; i++){
      arr[i-1] = clust_seq.exclusive_dmerge(i)/jet_scale;
    }
    // Save in Moment Map
    themap->set_jet_moment_array(mom_pos,(*jit),arr);
  }

}

void YSplitterTool::finalize(){};


}  // namespace SpartyJet
}
